<?php
/**
 * The Conference Pro Custom functions and definitions
 *
 * @package The_Conference_Pro
 */

/**
 * Show/Hide Admin Bar in frontend.
*/
if( ! get_theme_mod( 'ed_adminbar', true ) ) add_filter( 'show_admin_bar', '__return_false' );

if ( ! function_exists( 'the_conference_pro_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function the_conference_pro_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Conference Pro, use a find and replace
	 * to change 'the-conference-pro' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'the-conference-pro', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
    $menus = array(
        'primary'   => esc_html__( 'Primary', 'the-conference-pro' ),
    );

    if( is_polylang_active() ){
        $menus['language'] = esc_html__( 'Language', 'the-conference-pro' ); 
    }

	register_nav_menus( $menus );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-list',
		'gallery',
		'caption',
	) );
    
    // Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'the_conference_pro_custom_background_args', array(
		'default-color' => 'ffffff',
		'default-image' => '',
	) ) );
    
	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support( 
        'custom-logo', 
        array( 
            'width'       => 185,
            'height'      => 80,
            'flex-height' => true,
            'flex-width'  => true,
            'header-text' => array( 'site-title', 'site-description' ) 
        ) 
    );
    
    /**
     * Add support for custom header.
    */
    add_theme_support( 'custom-header', apply_filters( 'the_conference_pro_custom_header_args', array(
		'default-image' => get_template_directory_uri() . '/images/banner-img.jpg',
        'video'         => true,
		'width'         => 1920,
		'height'        => 1008, 
		'header-text'   => false
	) ) );

    // Register default headers.
    register_default_headers( array(
        'default-banner' => array(
            'url'           => '%s/images/banner-img.jpg',
            'thumbnail_url' => '%s/images/banner-img.jpg',
            'description'   => esc_html_x( 'Default Banner', 'header image description', 'the-conference-pro' ),
        ),
    ) );
 
    /**
     * Add Custom Images sizes.
    */    
    add_image_size( 'the-conference-banner-slider', 1920, 1008, true );
    add_image_size( 'the-conference-featured-page', 470, 470, true );
    add_image_size( 'the-conference-icon-text-image', 585, 550, true );
    add_image_size( 'the-conference-speaker', 384, 467, true );
    add_image_size( 'the-conference-gallery', 768, 768, true );
    add_image_size( 'the-conference-experience', 550, 350, true );
    add_image_size( 'the-conference-blog', 768, 519, true );
    add_image_size( 'the-conference-blog-fullwidth', 1320, 519, true );
    add_image_size( 'the-conference-related', 110, 83, true );
    add_image_size( 'the-conference-schema', 600, 60, true );    
    
    /** Starter Content */
    $starter_content = array(
        // Specify the core-defined pages to create and add custom thumbnails to some of them.
		'posts' => array( 
            'home', 
            'blog',
            'about' => array(
                'post_type'  => 'page',
                'post_title' => __( 'About Us', 'the-conference-pro' ),
                'template' => 'templates/about.php',
            ),
            'speaker' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Speakers', 'the-conference-pro' ),
                'template' => 'templates/speaker.php',
            ),
            'event-schedule' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Event schedule', 'the-conference-pro' ),
                'template' => 'templates/schedule.php',
            ),
            'ticket' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Buy Ticket', 'the-conference-pro' ),
                'template' => 'templates/ticket.php',
            ), 
            'testimonial' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Testimonials', 'the-conference-pro' ),
                'template' => 'templates/testimonial.php',
            ),
            'sponsor' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Sponsors', 'the-conference-pro' ),
                'template' => 'templates/sponsor.php',
            ),
            'faq' => array(
                'post_type'  => 'page',
                'post_title' => __( 'FAQ\'S', 'the-conference-pro' ),
                'template' => 'templates/faq.php',
            ),
            'gallery' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Our Gallery', 'the-conference-pro' ),
                'template' => 'templates/gallery.php',
            ),
            'contact' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Contact Us', 'the-conference-pro' ),
                'template' => 'templates/contact.php',
            ),
            'venue' => array(
                'post_type'  => 'page',
                'post_title' => __( 'Venue', 'the-conference-pro' ),
                'template' => 'templates/venue.php',
            ),
        ),
		
        // Default to a static front page and assign the front and posts pages.
		'options' => array(
			'show_on_front' => 'page',
			'page_on_front' => '{{home}}',
			'page_for_posts' => '{{blog}}',
		),
        
        // Set up nav menus for each of the two areas registered in the theme.
		'nav_menus' => array(
			// Assign a menu to the "top" location.
			'primary' => array(
				'name' => __( 'Primary', 'the-conference-pro' ),
				'items' => array(
					'page_home',
					'page_blog',
                    'page_about' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{about-us}}',
                    ),
                    'page_speaker' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{speaker}}',
                    ),
                    'page_schedule' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{event-schedule}}',
                    ),
                    'page_ticket' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{ticket}}',
                    ),
                    'page_testimonial' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{testimonial}}',
                    ),
                    'page_sponsor' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{sponsor}}',
                    ),
                    'page_faq' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{faq}}',
                    ),
                    'page_gallery' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{gallery}}',
                    ),
                    'page_contact' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{contact-us}}',
                    ),
                    'page_venu' => array(
                        'type'      => 'post_type',
                        'object'    => 'page',
                        'object_id' => '{{venue}}',
                    ),
				)
			)
		),
    );
    
    $starter_content = apply_filters( 'the_conference_pro_starter_content', $starter_content );

	add_theme_support( 'starter-content', $starter_content );
    
    // Add theme support for Responsive Videos.
    add_theme_support( 'jetpack-responsive-videos' );

    // Add excerpt support for pages
    add_post_type_support( 'page', 'excerpt' );

    // Remove widget block.
    remove_theme_support( 'widgets-block-editor' );

}
endif;
add_action( 'after_setup_theme', 'the_conference_pro_setup' );

if( ! function_exists( 'the_conference_pro_content_width' ) ) :
/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function the_conference_pro_content_width() {
	/** 
     * content width.
    */
    $GLOBALS['content_width'] = apply_filters( 'the_conference_pro_content_width', 640 );
}
endif;
add_action( 'after_setup_theme', 'the_conference_pro_content_width', 0 );

if( ! function_exists( 'the_conference_pro_template_redirect_content_width' ) ) :
/**
* Adjust content_width value according to template.
*
* @return void
*/
function the_conference_pro_template_redirect_content_width(){
	$sidebar = the_conference_pro_sidebar();
    if( $sidebar ){	   
        /** 
         * content width when sidebar is active.
        */
        $GLOBALS['content_width'] = 640;       
	}else{
        /** 
         * content width in singular page.
        */
        if( is_singular() ){
            if( the_conference_pro_sidebar( true ) === 'full-width-centered' ){
                $GLOBALS['content_width'] = 640;
            }else{
                $GLOBALS['content_width'] = 1320;                
            }                
        }else{
            $GLOBALS['content_width'] = 1320;
        }
	}
}
endif;
add_action( 'template_redirect', 'the_conference_pro_template_redirect_content_width' );

if( ! function_exists( 'the_conference_pro_scripts' ) ) :
/**
 * Enqueue scripts and styles.
 */
function the_conference_pro_scripts() {
	// Use minified libraries if SCRIPT_DEBUG is false
    $build                 = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '/build' : '';
    $suffix                = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
    $mcustomscrollbar_suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.concat.min';
    $ed_widget_sticky      = get_theme_mod( 'ed_last_widget_sticky', false );
    $ed_one_page           = get_theme_mod( 'ed_one_page', false );
    $ed_sticky_header      = get_theme_mod( 'ed_sticky_header', false );
    $lazy_load             = get_theme_mod( 'ed_lazy_load', true );
    $lazy_load_cimage      = get_theme_mod( 'ed_lazy_load_cimage', true );
    $ed_animate            = get_theme_mod( 'ed_animate', true );
    $ed_googlefont_local   = get_theme_mod( 'ed_googlefont_local', false );
    $time_zone             = get_theme_mod( 'timer_time_zone', 'UTC' );
    $ed_banner_event_timer = get_theme_mod( 'ed_banner_event_timer', true );
    $banner_event_timer    = get_theme_mod( 'banner_event_timer', '2022-08-20' );
    $banner_event_time     = get_theme_mod( 'banner_event_time', '00:00' );
    $ed_cta_event_timer    = get_theme_mod( 'ed_timer_cta_event_time', true );
    $cta_event_timer       = get_theme_mod( 'timer_cta_event_time', '2022-08-20' );
    $cta_event_time        = get_theme_mod( 'cta_event_time', '00:00' );
    $ed_cta_about_timer    = get_theme_mod( 'ed_about_event_timer', true );
    $cta_about_timer       = get_theme_mod( 'about_timer_cta_event_time', '2022-08-20' );
    $cta_about_time        = get_theme_mod( 'about_cta_event_time', '00:00' );

    if( ! $ed_googlefont_local )
    wp_enqueue_style( 'the-conference-pro-google-fonts', the_conference_pro_fonts_url(), array(), null );

    if( is_woocommerce_activated() )
    wp_enqueue_style( 'the-conference-pro-woocommerce', get_template_directory_uri(). '/css' . $build . '/woocommerce' . $suffix . '.css', array(), THE_CONFERENCE_PRO_THEME_VERSION );
    
    if ( is_front_page() ) 
    wp_enqueue_style( 'owl-carousel', get_template_directory_uri(). '/css' . $build . '/owl.carousel' . $suffix . '.css', array(), '2.2.1' );

    wp_enqueue_style( 'jquery-mcustomscrollbar', get_template_directory_uri(). '/css' . $build . '/jquery.mCustomScrollbar' . $suffix . '.css', array(), '3.1.5' );    

    wp_enqueue_style( 'the-conference-pro', get_stylesheet_uri(), array(), THE_CONFERENCE_PRO_THEME_VERSION );
    
    if( ( $lazy_load || $lazy_load_cimage ) && ! is_page_template( 'templates/portfolio.php' ) ){
        wp_enqueue_script( 'layzr', get_template_directory_uri() . '/js/layzr.min.js', array('jquery'), '2.0.4', true );
    }
    
    //Fancy Box
    if( get_theme_mod( 'ed_lightbox', true ) ){
        wp_enqueue_style( 'jquery-fancybox', get_template_directory_uri() . '/css' . $build . '/jquery.fancybox' . $suffix . '.css', array(), '3.5.7' );
        wp_enqueue_script( 'jquery-fancybox', get_template_directory_uri() . '/js' . $build . '/jquery.fancybox' . $suffix . '.js', array('jquery'), '3.5.7', true );
    }

    if( $ed_animate ){
        wp_enqueue_style( 'animate', get_template_directory_uri(). '/css' . $build . '/animate' . $suffix . '.css', array(), '3.5.2' );
        wp_enqueue_script( 'wow', get_template_directory_uri() . '/js' . $build . '/wow' . $suffix . '.js', array( 'jquery' ), '2.0.3', true );
    }

    if( $ed_one_page ){
        wp_enqueue_script( 'jquery-nav', get_template_directory_uri() . '/js' . $build . '/jquery.nav' . $suffix . '.js', array( 'jquery' ), '3.0.0', true );
    }
 
    if( $ed_widget_sticky ){
        wp_enqueue_script( 'sticky-kit', get_template_directory_uri() . '/js' . $build . '/sticky-kit' . $suffix . '.js', array( 'jquery' ), '1.1.3', true );
    }
    
    if( ( is_front_page() && ! is_home() && ( $ed_banner_event_timer || $ed_cta_event_timer ) ) || ( is_page_template( 'templates/about.php' ) && $ed_cta_about_timer ) ){
        wp_enqueue_script( 'moment-timezone-with-data', get_template_directory_uri() . '/js' . $build . '/moment-timezone-with-data' . $suffix . '.js', array( 'jquery', 'moment' ), '2.2.0', true );
        wp_enqueue_script( 'jquery-countdown', get_template_directory_uri() . '/js' . $build . '/jquery.countdown' . $suffix . '.js', array( 'jquery' ), '2.2.0', true );
    }

    wp_enqueue_script( 'all', get_template_directory_uri() . '/js' . $build . '/all' . $suffix . '.js', array( 'jquery' ), '5.6.3', true );
    wp_enqueue_script( 'v4-shims', get_template_directory_uri() . '/js' . $build . '/v4-shims' . $suffix . '.js', array( 'jquery' ), '5.6.3', true );

    if ( is_front_page() ) 
    wp_enqueue_script( 'owl-carousel', get_template_directory_uri() . '/js' . $build . '/owl.carousel' . $suffix . '.js', array( 'jquery' ), '2.2.1', true );
    
    wp_enqueue_script( 'jquery-mousewheel', get_template_directory_uri() . '/js' . $build . '/jquery.mousewheel' . $suffix . '.js', array(), '3.0.6', true );
    wp_enqueue_script( 'jquery-mcustomscrollbar', get_template_directory_uri() . '/js' . $build . '/jquery.mCustomScrollbar' . $mcustomscrollbar_suffix . '.js', array( 'jquery' ), '3.1.5', true );
	wp_enqueue_script( 'the-conference-pro', get_template_directory_uri() . '/js' . $build . '/custom' . $suffix . '.js', array( 'jquery', 'jquery-ui-tabs', 'jquery-ui-core' ), THE_CONFERENCE_PRO_THEME_VERSION, true );

    $array = array( 
        'rtl'                => is_rtl(),
        'auto'               => get_theme_mod( 'slider_auto', true ),
        'loop'               => get_theme_mod( 'slider_loop', true ),
        'animation'          => get_theme_mod( 'slider_animation' ),
        'h_layout'           => get_theme_mod( 'header_layout', 'one' ),
        'drop_cap'           => get_theme_mod( 'ed_drop_cap', true ),
        'singular'           => is_singular(),
        'sticky_header'      => $ed_sticky_header,
        'sticky_widget'      => $ed_widget_sticky,
        'event_expired'      => esc_html__( 'Event Expired', 'the-conference-pro' ),
        'time_zone'          => $time_zone,
        'ed_banner_timer'    => $ed_banner_event_timer,
        'banner_event_timer' => (string) ( $banner_event_timer . ' '. $banner_event_time ),
        'ed_cta_timer'       => $ed_cta_event_timer,
        'cta_event_timer'    => (string) ( $cta_event_timer . ' ' . $cta_event_time ),
        'ed_abotu_cta_timer' => $ed_cta_about_timer,
        'about_event_timer'  => (string) ( $cta_about_timer . ' ' . $cta_about_time ),
        'lightbox'           => get_theme_mod( 'ed_lightbox', true ),
        'animate'            => $ed_animate,
        'onepage'            => get_theme_mod( 'ed_one_page', false ),
    );
    
    wp_localize_script( 'the-conference-pro', 'the_conference_pro_data', $array );
    
    /** Ajax Pagination */
    $pagination = get_theme_mod( 'pagination_type', 'numbered' );
    $loadmore   = get_theme_mod( 'load_more_label', __( 'Load More Posts', 'the-conference-pro' ) );
    $loading    = get_theme_mod( 'loading_label', __( 'Loading...', 'the-conference-pro' ) );
    $nomore     = get_theme_mod( 'no_more_label', __( 'No More Post', 'the-conference-pro' ) );
       
    // Add parameters for the JS
    global $wp_query;
    $max = $wp_query->max_num_pages;
    $paged = ( get_query_var( 'paged' ) > 1 ) ? get_query_var( 'paged' ) : 1;
    
    wp_enqueue_script( 'the-conference-pro-ajax', get_template_directory_uri() . '/js' . $build . '/ajax' . $suffix . '.js', array('jquery'), THE_CONFERENCE_PRO_THEME_VERSION, true );
    
    wp_localize_script( 
        'the-conference-pro-ajax', 
        'the_conference_pro_ajax',
        array(
            'url'        => admin_url( 'admin-ajax.php' ),
            'startPage'  => $paged,
            'maxPages'   => $max,
            'nextLink'   => next_posts( $max, false ),
            'autoLoad'   => $pagination,
            'loadmore'   => $loadmore,
            'loading'    => $loading,
            'nomore'     => $nomore,
            'plugin_url' => plugins_url(),                
         )
    );
    
    if ( is_jetpack_activated( true ) ) {
        wp_enqueue_style( 'tiled-gallery', plugins_url() . '/jetpack/modules/tiled-gallery/tiled-gallery/tiled-gallery.css' );            
    }
    
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
endif;
add_action( 'wp_enqueue_scripts', 'the_conference_pro_scripts' );

if( ! function_exists( 'the_conference_pro_admin_scripts' ) ) :
/**
 * Enqueue admin scripts and styles.
*/
function the_conference_pro_admin_scripts( $hook ){    

    // Add custom widget script
    wp_register_script( 'the-conference-pro-custom-widget-js', get_template_directory_uri() . '/inc/js/custom-widget.js',  array( 'jquery', 'jquery-ui-sortable' ), THE_CONFERENCE_PRO_THEME_VERSION ); 

    $confirming = array( 'are_you_sure' => esc_html__( 'Are you sure?', 'the-conference-pro' ) );

    wp_localize_script( 'the-conference-pro-custom-widget-js', 'confirming', $confirming );

    wp_enqueue_script( 'the-conference-pro-custom-widget-js' );

    // if( $hook == 'post-new.php' || $hook == 'post.php' || $hook == 'user-new.php' || $hook == 'user-edit.php' || $hook == 'profile.php' ){
        wp_enqueue_style( 'the-conference-pro-admin', get_template_directory_uri() . '/inc/css/admin.css', '', THE_CONFERENCE_PRO_THEME_VERSION );
        wp_enqueue_script( 'the-conference-pro-admin', get_template_directory_uri() . '/inc/js/admin.js', array( 'jquery', 'jquery-ui-sortable' ), THE_CONFERENCE_PRO_THEME_VERSION, false );
    // }
}
endif; 
add_action( 'admin_enqueue_scripts', 'the_conference_pro_admin_scripts' );

if( ! function_exists( 'the_conference_pro_body_classes' ) ) :
/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function the_conference_pro_body_classes( $classes ) {
    $banner_control   = get_theme_mod( 'ed_banner_section', 'static_banner' );
    $header_image     = get_header_image_tag(); // get custom header image tag
    $blog_layout      = get_theme_mod( 'blog_page_layout', 'classic-view' );
    $bg_pattern       = get_theme_mod( 'bg_pattern', 'nobg' );
    $bg               = get_theme_mod( 'body_bg', 'image' );

    // Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}
    
    if( is_front_page() && ! is_home() && 'no_banner' != $banner_control && ( has_header_video() ||  ! empty( $header_image ) ) ){
        $classes[] = 'homepage hasbanner';
    }

    // Adds a class of custom-background-image to sites with a custom background image.
	if( $bg == 'pattern' && $bg_pattern != 'nobg' ){
		$classes[] = 'custom-background-pattern custom-background';
	}
    
    if( $bg == 'image' && get_background_image() ) {
		$classes[] = 'custom-background-image';
	}
    
    // Adds a class of custom-background-color to sites with a custom background color.
    if( get_background_color() != 'ffffff' ) {
		$classes[] = 'custom-background-color';
	}
    
    $classes[] = the_conference_pro_sidebar( true );

    if( is_home() ){
        switch( $blog_layout ){
            case 'classic-view':
            $classes[] = 'classic-view';
            break;
            case 'list-view':
            $classes[] = 'list-view';
            break;
        }
    }

    if( is_archive() || is_search() ){
        $classes[] = 'list-view';
    }

	return $classes;
}
endif;
add_filter( 'body_class', 'the_conference_pro_body_classes' );

if( ! function_exists( 'the_conference_pro_post_classes' ) ) :
/**
 * Add custom classes to the array of post classes.
*/
function the_conference_pro_post_classes( $classes ){
    $enable_social_share = get_theme_mod( 'ed_social_sharing', true );

    $classes[] = 'latest_post';

    if( is_single() && $enable_social_share ){
        $classes[] = 'sticky-social';
    }
    
    return $classes;
}
endif;
add_filter( 'post_class', 'the_conference_pro_post_classes' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function the_conference_pro_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'the_conference_pro_pingback_header' );

if( ! function_exists( 'the_conference_pro_change_comment_form_default_fields' ) ) :
/**
 * Change Comment form default fields i.e. author, email & url.
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function the_conference_pro_change_comment_form_default_fields( $fields ){    
    // get the current commenter if available
    $commenter = wp_get_current_commenter();
 
    // core functionality
    $req      = get_option( 'require_name_email' );
    $aria_req = ( $req ? " aria-required='true'" : '' );
    $required = ( $req ? " required" : '' );
    $author   = ( $req ? __( 'Name*', 'the-conference-pro' ) : __( 'Name', 'the-conference-pro' ) );
    $email    = ( $req ? __( 'Email*', 'the-conference-pro' ) : __( 'Email', 'the-conference-pro' ) );
 
    // Change just the author field
    $fields['author'] = '<p class="comment-form-author"><label class="screen-reader-text" for="author">' . esc_html__( 'Name', 'the-conference-pro' ) . '<span class="required">*</span></label><input id="author" name="author" placeholder="' . esc_attr( $author ) . '" type="text" value="' . esc_attr( $commenter['comment_author'] ) . '" size="30"' . $aria_req . $required . ' /></p>';
    
    $fields['email'] = '<p class="comment-form-email"><label class="screen-reader-text" for="email">' . esc_html__( 'Email', 'the-conference-pro' ) . '<span class="required">*</span></label><input id="email" name="email" placeholder="' . esc_attr( $email ) . '" type="text" value="' . esc_attr(  $commenter['comment_author_email'] ) . '" size="30"' . $aria_req . $required. ' /></p>';
    
    $fields['url'] = '<p class="comment-form-url"><label class="screen-reader-text" for="url">' . esc_html__( 'Website', 'the-conference-pro' ) . '</label><input id="url" name="url" placeholder="' . esc_attr__( 'Website', 'the-conference-pro' ) . '" type="text" value="' . esc_attr( $commenter['comment_author_url'] ) . '" size="30" /></p>'; 
    
    return $fields;    
}
endif;
add_filter( 'comment_form_default_fields', 'the_conference_pro_change_comment_form_default_fields' );

if( ! function_exists( 'the_conference_pro_change_comment_form_defaults' ) ) :
/**
 * Change Comment Form defaults
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function the_conference_pro_change_comment_form_defaults( $defaults ){    
    $defaults['comment_field'] = '<p class="comment-form-comment"><label class="screen-reader-text" for="comment">' . esc_html__( 'Comment', 'the-conference-pro' ) . '</label><textarea id="comment" name="comment" placeholder="' . esc_attr__( 'Comment', 'the-conference-pro' ) . '" cols="45" rows="8" aria-required="true" required></textarea></p>';
    
    return $defaults;    
}
endif;
add_filter( 'comment_form_defaults', 'the_conference_pro_change_comment_form_defaults' );

if ( ! function_exists( 'the_conference_pro_excerpt_more' ) ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... * 
 */
function the_conference_pro_excerpt_more( $more ) {
	return is_admin() ? $more : ' &hellip; ';
}

endif;
add_filter( 'excerpt_more', 'the_conference_pro_excerpt_more' );

if ( ! function_exists( 'the_conference_pro_excerpt_length' ) ) :
/**
 * Changes the default 55 character in excerpt 
*/
function the_conference_pro_excerpt_length( $length ) {
	$excerpt_length = get_theme_mod( 'excerpt_length', 55 );
    return is_admin() ? $length : absint( $excerpt_length );    
}
endif;
add_filter( 'excerpt_length', 'the_conference_pro_excerpt_length', 999 );

if( ! function_exists( 'the_conference_pro_get_the_archive_title' ) ) :
/**
 * Filter Archive Title
*/
function the_conference_pro_get_the_archive_title( $title ){
    $ed_prefix = get_theme_mod( 'ed_prefix_archive', false );
    
    if( is_category() ){
        if( $ed_prefix ){
            $title = '<h1 class="page-title">' . single_cat_title( '', false ) . '</h1>';
        }else{
            /* translators: Category archive title. 1: Category name */
            $title = sprintf( __( '%1$sCategory%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . single_cat_title( '', false ) . '</h1>' );
        }
    }elseif ( is_tag() ){
        if( $ed_prefix ){
            $title = '<h1 class="page-title">' . single_tag_title( '', false ) . '</h1>';    
        }else{
            /* translators: Tag archive title. 1: Tag name */
            $title = sprintf( __( '%1$sTag%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . single_tag_title( '', false ) . '</h1>' );
        }
    }elseif ( is_year() ) {
        if( $ed_prefix ){
            $title = '<h1 class="page-title">' . get_the_date( _x( 'Y', 'yearly archives date format', 'the-conference-pro' ) ) . '</h1>';
        }else{
            /* translators: Yearly archive title. 1: Year */
            $title = sprintf( __( '%1$sYear%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . get_the_date( _x( 'Y', 'yearly archives date format', 'the-conference-pro' ) ) . '</h1>' );
        }
    }elseif ( is_month() ) {
        if( $ed_prefix ){
            $title = '<h1 class="page-title">' . get_the_date( _x( 'F Y', 'monthly archives date format', 'the-conference-pro' ) ) . '</h1>';
        }else{
            /* translators: Monthly archive title. 1: Month name and year */
            $title = sprintf( __( '%1$sMonth%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . get_the_date( _x( 'F Y', 'monthly archives date format', 'the-conference-pro' ) ) . '</h1>' );
        }
    }elseif ( is_day() ) {
       if( $ed_prefix ){
            $title = '<h1 class="page-title">' . get_the_date( _x( 'F j, Y', 'daily archives date format', 'the-conference-pro' ) ) . '</h1>';
        }else{
            /* translators: Daily archive title. 1: Date */
            $title = sprintf( __( '%1$sDay%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . get_the_date( _x( 'F j, Y', 'daily archives date format', 'the-conference-pro' ) ) . '</h1>' );
        }
    }elseif ( is_post_type_archive() ) {
        if( is_post_type_archive( 'product' ) ){
            $title = '<h1 class="page-title">' . get_the_title( get_option( 'woocommerce_shop_page_id' ) ) . '</h1>';
        }else{
            if( $ed_prefix ){
                $title = '<h1 class="page-title">' . post_type_archive_title( '', false ) . '</h1>';
            }else{
                /* translators: Post type archive title. 1: Post type name */
                $title = sprintf( __( '%1$sArchives%2$s %3$s', 'the-conference-pro' ), '<span class="sub-title">', '</span>', '<h1 class="page-title">' . post_type_archive_title( '', false ) . '</h1>' );
            }
        }
    }elseif ( is_tax() ) {
        $tax = get_taxonomy( get_queried_object()->taxonomy );
        if( $ed_prefix ){
            $title = '<h1 class="page-title">' . single_term_title( '', false ) . '</h1>';
        }else{                                                            
            /* translators: Taxonomy term archive title. 1: Taxonomy singular name, 2: Current taxonomy term */
            $title = sprintf( __( '%1$s: %2$s', 'the-conference-pro' ), '<span class="sub-title">' . $tax->labels->singular_name . '</span>', '<h1 class="page-title">' . single_term_title( '', false ) . '</h1>' );
        }
    }
        
    return $title;
    
}
endif;
add_filter( 'get_the_archive_title', 'the_conference_pro_get_the_archive_title' );

if( ! function_exists( 'the_conference_pro_remove_archive_description' ) ) :
/**
 * filter the_archive_description & get_the_archive_description to show post type archive
 * @param  string $description original description
 * @return string post type description if on post type archive
 */
function the_conference_pro_remove_archive_description( $description ){
    $ed_shop_archive_description = get_theme_mod( 'ed_shop_archive_description', false );
    $shop_archive_description    = get_theme_mod( 'shop_archive_description' );
    if( is_post_type_archive( 'product' ) ) {
        if( ! $ed_shop_archive_description ){
            $description = '';
        }else{
            if( $shop_archive_description ) $description = $shop_archive_description;
        }
    }
    return wpautop( wp_kses_post( $description ) );
}
endif;
add_filter( 'get_the_archive_description', 'the_conference_pro_remove_archive_description' );

if( ! function_exists( 'the_conference_pro_migrate_free_option' ) ) :
/**
 * Function to migrate free theme option to pro theme
*/
function the_conference_pro_migrate_free_option(){    
    $fresh = get_option( '_the_conference_pro_fresh_install' ); //flag to check if it is first switch
    $pages = array(
        'about' => array( 
            'page_name'     => esc_html__( 'About Us', 'the-conference-pro' ),
            'page_template' => 'templates/about.php'
        ),
        'speaker' => array( 
            'page_name'     => esc_html__( 'Speakers', 'the-conference-pro' ),
            'page_template' => 'templates/speaker.php'
        ),
        'schedule' => array(
            'page_name' => __( 'Event schedule', 'the-conference-pro' ),
            'template' => 'templates/schedule.php',
        ),
        'ticket' => array( 
            'page_name'     => esc_html__( 'Buy Ticket', 'the-conference-pro' ),
            'page_template' => 'templates/ticket.php'
        ),
        'testimonial' => array( 
            'page_name'     => esc_html__( 'Testimonials', 'the-conference-pro' ),
            'page_template' => 'templates/testimonial.php'
        ),
        'sponsor' => array(
            'page_name' => __( 'Sponsors', 'the-conference-pro' ),
            'template' => 'templates/sponsor.php',
        ),
        'faq' => array( 
            'page_name'     => esc_html__( 'FAQ\'S', 'the-conference-pro' ),
            'page_template' => 'templates/faq.php'
        ),
        'gallery' => array(
            'page_name' => __( 'Our Gallery', 'the-conference-pro' ),
            'template' => 'templates/gallery.php',
        ),
        'contact' => array( 
            'page_name'     => esc_html__( 'Contact Us', 'the-conference-pro' ),
            'page_template' => 'templates/contact.php'
        ),
        'venue' => array(
            'page_name' => __( 'Venue', 'the-conference-pro' ),
            'template' => 'templates/venue.php',
        ),
    );
    
    if( ! $fresh ){        
        $options = get_option( 'theme_mods_conference' );
        
        if( $options ){
            //migrate free theme option to pro
            foreach( $options as $option => $value ){
                if( $option !== 'sidebars_widgets' ){
                    set_theme_mod( $option, $value );
                }    
            }
            //create default pages on theme migration
            foreach( $pages as $page => $val ){
                the_conference_pro_create_post( $val['page_name'], $page, $val['page_template'] );
            } 
        }        
        update_option( '_the_conference_pro_fresh_install', true );  
    }
}
endif;
add_action( 'after_switch_theme', 'the_conference_pro_migrate_free_option' );

if( ! function_exists( 'the_conference_pro_single_post_schema' ) ) :
/**
 * Single Post Schema
 *
 * @return string
 */
function the_conference_pro_single_post_schema() {
    if ( is_singular( 'post' ) ) {
        global $post;
        $custom_logo_id = get_theme_mod( 'custom_logo' );

        $site_logo   = wp_get_attachment_image_src( $custom_logo_id , 'the-conference-schema' );
        $images      = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'full' );
        $excerpt     = the_conference_pro_escape_text_tags( $post->post_excerpt );
        $content     = $excerpt === "" ? mb_substr( the_conference_pro_escape_text_tags( $post->post_content ), 0, 110 ) : $excerpt;
        $schema_type = ! empty( $custom_logo_id ) && has_post_thumbnail( $post->ID ) ? "BlogPosting" : "Blog";

        $args = array(
            "@context"  => "https://schema.org",
            "@type"     => $schema_type,
            "mainEntityOfPage" => array(
                "@type" => "WebPage",
                "@id"   => esc_url( get_permalink( $post->ID ) )
            ),
            "headline"  => esc_html( get_the_title( $post->ID ) ),
            "datePublished" => esc_html( get_the_time( DATE_ISO8601, $post->ID ) ),
            "dateModified"  => esc_html( get_post_modified_time(  DATE_ISO8601, __return_false(), $post->ID ) ),
            "author"        => array(
                "@type"     => "Person",
                "name"      => the_conference_pro_escape_text_tags( get_the_author_meta( 'display_name', $post->post_author ) )
            ),
            "description" => ( class_exists('WPSEO_Meta') ? WPSEO_Meta::get_value( 'metadesc' ) : $content )
        );

        if ( has_post_thumbnail( $post->ID ) ) :
            $args['image'] = array(
                "@type"  => "ImageObject",
                "url"    => $images[0],
                "width"  => $images[1],
                "height" => $images[2]
            );
        endif;

        if ( ! empty( $custom_logo_id ) ) :
            $args['publisher'] = array(
                "@type"       => "Organization",
                "name"        => esc_html( get_bloginfo( 'name' ) ),
                "description" => wp_kses_post( get_bloginfo( 'description' ) ),
                "logo"        => array(
                    "@type"   => "ImageObject",
                    "url"     => $site_logo[0],
                    "width"   => $site_logo[1],
                    "height"  => $site_logo[2]
                )
            );
        endif;

        echo '<script type="application/ld+json">';
        if ( version_compare( PHP_VERSION, '5.4.0' , '>=' ) ) {
            echo wp_json_encode( $args, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
        } else {
            echo wp_json_encode( $args );
        }
        echo '</script>';
    }
}
endif;
add_action( 'wp_head', 'the_conference_pro_single_post_schema' );

if( ! function_exists( 'the_conference_pro_google_analytics' ) ) :
/**
 * Place to add Google Analytics Code
 */
function the_conference_pro_google_analytics() {
    $google_ad_code = get_theme_mod( 'google_analytics_ad_code' );
    if( $google_ad_code ) echo $google_ad_code;
}
endif;
add_action( 'wp_head' , 'the_conference_pro_google_analytics' );

if( ! function_exists( 'the_conference_pro_get_comment_author_link' ) ) :
/**
 * Filter to modify comment author link
 * @link https://developer.wordpress.org/reference/functions/get_comment_author_link/
 */
function the_conference_pro_get_comment_author_link( $return, $author, $comment_ID ){
    $comment = get_comment( $comment_ID );
    $url     = get_comment_author_url( $comment );
    $author  = get_comment_author( $comment );
 
    if ( empty( $url ) || 'http://' == $url )
        $return = '<span itemprop="name">'. esc_html( $author ) .'</span>';
    else
        $return = '<span itemprop="name"><a href=' . esc_url( $url ) . ' rel="external nofollow noopener" class="url" itemprop="url">' . esc_html( $author ) . '</a></span>';

    return $return;
}
endif;
add_filter( 'get_comment_author_link', 'the_conference_pro_get_comment_author_link', 10, 3 );

if ( !function_exists( 'the_conference_pro_video_controls' ) ) :
/**
 * Customize video play/pause button in the custom header.
 *
 * @param array $settings Video settings.
 */
function the_conference_pro_video_controls( $settings ) {
    $settings['l10n']['play'] = '<span class="screen-reader-text">' . __( 'Play background video', 'the-conference-pro' ) . '</span>' . the_conference_pro_get_svg( array( 'icon' => 'play' ) );
    $settings['l10n']['pause'] = '<span class="screen-reader-text">' . __( 'Pause background video', 'the-conference-pro' ) . '</span>' . the_conference_pro_get_svg( array( 'icon' => 'pause' ) );
    return $settings;
}
endif;
add_filter( 'header_video_settings', 'the_conference_pro_video_controls' );

if( ! function_exists( 'the_conference_pro_include_svg_icons' ) ) :
/**
 * Add SVG definitions to the footer.
 */
function the_conference_pro_include_svg_icons() {
    // Define SVG sprite file.
    $svg_icons = get_parent_theme_file_path( '/images/svg-icons.svg' );

    // If it exists, include it.
    if ( file_exists( $svg_icons ) ) {
        require_once( $svg_icons );
    }
}
endif;
add_action( 'wp_footer', 'the_conference_pro_include_svg_icons', 9999 );


if( ! function_exists( 'the_conference_pro_media_library_custom_image_size' ) ) :
/**
 * Add custom image size to media library.
 */
function the_conference_pro_media_library_custom_image_size( $sizes ) {
    return array_merge( $sizes, array(
        'the-conference-experience' => __( 'Experience Image', 'the-conference-pro' ),
        'the-conference-gallery'    => __( 'Gallery Image', 'the-conference-pro' ),
    ) );
}
endif;
add_filter( 'image_size_names_choose', 'the_conference_pro_media_library_custom_image_size' );

if( ! function_exists( 'the_conference_pro_filter_post_gallery' ) ) :
/**
 * Filter the output of the gallery. 
*/ 
function the_conference_pro_filter_post_gallery( $output, $attr, $instance ){
    global $post, $wp_locale;

    $html5 = current_theme_supports( 'html5', 'gallery' );
    $atts = shortcode_atts( array(
    	'order'      => 'ASC',
    	'orderby'    => 'menu_order ID',
    	'id'         => $post ? $post->ID : 0,
    	'itemtag'    => $html5 ? 'figure'     : 'dl',
    	'icontag'    => $html5 ? 'div'        : 'dt',
    	'captiontag' => $html5 ? 'figcaption' : 'dd',
    	'columns'    => 3,
    	'size'       => 'thumbnail',
    	'include'    => '',
    	'exclude'    => '',
    	'link'       => ''
    ), $attr, 'gallery' );
    
    $id = intval( $atts['id'] );
    
    if ( ! empty( $atts['include'] ) ) {
    	$_attachments = get_posts( array( 'include' => $atts['include'], 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    
    	$attachments = array();
    	foreach ( $_attachments as $key => $val ) {
    		$attachments[$val->ID] = $_attachments[$key];
    	}
    } elseif ( ! empty( $atts['exclude'] ) ) {
    	$attachments = get_children( array( 'post_parent' => $id, 'exclude' => $atts['exclude'], 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    } else {
    	$attachments = get_children( array( 'post_parent' => $id, 'post_status' => 'inherit', 'post_type' => 'attachment', 'post_mime_type' => 'image', 'order' => $atts['order'], 'orderby' => $atts['orderby'] ) );
    }
    
    if ( empty( $attachments ) ) {
    	return '';
    }
    
    if ( is_feed() ) {
    	$output = "\n";
    	foreach ( $attachments as $att_id => $attachment ) {
    		$output .= wp_get_attachment_link( $att_id, $atts['size'], true ) . "\n";
    	}
    	return $output;
    }
    
    $itemtag = tag_escape( $atts['itemtag'] );
    $captiontag = tag_escape( $atts['captiontag'] );
    $icontag = tag_escape( $atts['icontag'] );
    $valid_tags = wp_kses_allowed_html( 'post' );
    if ( ! isset( $valid_tags[ $itemtag ] ) ) {
    	$itemtag = 'dl';
    }
    if ( ! isset( $valid_tags[ $captiontag ] ) ) {
    	$captiontag = 'dd';
    }
    if ( ! isset( $valid_tags[ $icontag ] ) ) {
    	$icontag = 'dt';
    }
    
    $columns = intval( $atts['columns'] );
    $itemwidth = $columns > 0 ? floor(100/$columns) : 100;
    $float = is_rtl() ? 'right' : 'left';
    
    $selector = "gallery-{$instance}";
    
    $gallery_style = '';
    
    /**
     * Filter whether to print default gallery styles.
     *
     * @since 3.1.0
     *
     * @param bool $print Whether to print default gallery styles.
     *                    Defaults to false if the theme supports HTML5 galleries.
     *                    Otherwise, defaults to true.
     */
    if ( apply_filters( 'use_default_gallery_style', ! $html5 ) ) {
    	$gallery_style = "
    	<style type='text/css'>
    		#{$selector} {
    			margin: auto;
    		}
    		#{$selector} .gallery-item {
    			float: {$float};
    			margin-top: 10px;
    			text-align: center;
    			width: {$itemwidth}%;
    		}
    		#{$selector} img {
    			border: 2px solid #cfcfcf;
    		}
    		#{$selector} .gallery-caption {
    			margin-left: 0;
    		}
    		/* see gallery_shortcode() in wp-includes/media.php */
    	</style>\n\t\t";
    }
    
    $size_class = sanitize_html_class( $atts['size'] );
    $gallery_div = "<div id='$selector' class='gallery galleryid-{$id} gallery-columns-{$columns} gallery-size-{$size_class}'>";
    
    /**
     * Filter the default gallery shortcode CSS styles.
     *
     * @since 2.5.0
     *
     * @param string $gallery_style Default CSS styles and opening HTML div container
     *                              for the gallery shortcode output.
     */
    $output = apply_filters( 'gallery_style', $gallery_style . $gallery_div );
    
    $i = 0; 
    foreach ( $attachments as $id => $attachment ) {
            
    	$attr = ( trim( $attachment->post_excerpt ) ) ? array( 'aria-describedby' => "$selector-$id" ) : '';
    	if ( ! empty( $atts['link'] ) && 'file' === $atts['link'] ) {
    		//$image_output = wp_get_attachment_link( $id, $atts['size'], false, false, false, $attr ); // for attachment url 
            $image_output = "<a href='" . wp_get_attachment_url( $id ) . "' data-fancybox='group{$columns}' data-caption='" . esc_attr( $attachment->post_excerpt ) . "'>";
            $image_output .= "<img src='" . wp_get_attachment_image_url( $id, $atts['size'] ) . "' alt='" . trim( strip_tags( get_post_meta( $id, '_wp_attachment_image_alt', true ) ) ) . "' />";
            $image_output .= "</a>";
    	} elseif ( ! empty( $atts['link'] ) && 'none' === $atts['link'] ) {
    		$image_output = wp_get_attachment_image( $id, $atts['size'], false, $attr );
    	} else {
    		$image_output = wp_get_attachment_link( $id, $atts['size'], true, false, false, $attr ); //for attachment page
    	}
    	$image_meta  = wp_get_attachment_metadata( $id );
    
    	$orientation = '';
    	if ( isset( $image_meta['height'], $image_meta['width'] ) ) {
    		$orientation = ( $image_meta['height'] > $image_meta['width'] ) ? 'portrait' : 'landscape';
    	}
    	$output .= "<{$itemtag} class='gallery-item'>";
    	$output .= "
    		<{$icontag} class='gallery-icon {$orientation}'>
    			$image_output
    		</{$icontag}>";
    	if ( $captiontag && trim($attachment->post_excerpt) ) {
    		$output .= "
    			<{$captiontag} class='wp-caption-text gallery-caption' id='$selector-$id'>
    			" . wptexturize($attachment->post_excerpt) . "
    			</{$captiontag}>";
    	}
    	$output .= "</{$itemtag}>";
    	if ( ! $html5 && $columns > 0 && ++$i % $columns == 0 ) {
    		$output .= '<br style="clear: both" />';
    	}
    }
    
    if ( ! $html5 && $columns > 0 && $i % $columns !== 0 ) {
    	$output .= "
    		<br style='clear: both' />";
    }
    
    $output .= "
    	</div>\n";
    
    return $output;
}
endif;
if ( class_exists( 'Jetpack' ) ) {
    if( !Jetpack::is_module_active( 'carousel' ) ){
        add_filter( 'post_gallery', 'the_conference_pro_filter_post_gallery', 10, 3 );
    }
}else{
    add_filter( 'post_gallery', 'the_conference_pro_filter_post_gallery', 10, 3 );
}

if( ! function_exists( 'the_conference_pro_activate_notice' ) ) :
/**
 * Theme activation notice.
*/
function the_conference_pro_activate_notice() {
    global $current_user;
    $theme_list       = wp_get_theme( 'the-conference-pro' );
    $add_license      = get_option( 'the-conference-pro_license_key' );
    $current_screen   = get_current_screen();    
    $activate_license = get_option( 'the-conference-pro_license_key_status' );
    $statuses         = array( 'invalid', 'inactive', 'expired', 'disabled', 'site_inactive' );   
    
    if( $theme_list->exists() && ( empty( $add_license ) || in_array( $activate_license, $statuses ) ) && $current_screen->base != 'appearance_page_the-conference-pro-license' ) { ?>
        <div class="notice notice-info is-dismissible">
            <p><?php esc_html_e( 'Activate Theme License ( The Conference Pro ) to enjoy the full benefits of using the theme. We\'re sorry about this extra step but we built the activation to prevent mass piracy of our themes. This allows us to better serve our paying customers. An active theme comes with free updates, top notch support and guaranteed latest WordPress support.', 'the-conference-pro' ); ?>
            </p>
            <p>
                <span style="color:red;"><?php esc_html_e( 'Please Activate Theme License!', 'the-conference-pro' ); ?></span> - <a href="<?php echo esc_url( 'themes.php?page=the-conference-pro-license' ); ?>"><u><?php esc_html_e( 'Click here to enter your license key', 'the-conference-pro' ); ?></u></a> - <?php esc_html_e( 'if there is an error, please contact us at ', 'the-conference-pro' ); ?><a href="mailto:support@raratheme.com" target="_blank">support@raratheme.com</a> - <a href="<?php echo esc_url( 'https://rarathemes.com/activate-theme-license/' ); ?>" target="_blank"><u><?php esc_html_e( 'How to activate the theme license', 'the-conference-pro' ); ?></u></a>.
            </p> 
        </div>  
        <?php
    }
}
endif;
add_action( 'admin_notices', 'the_conference_pro_activate_notice' );

if( ! function_exists( 'the_conference_pro_redirect_on_activation' ) ) :
/**
 * Redirect to Getting Started page on theme activation
*/
function the_conference_pro_redirect_on_activation() {
    global $pagenow;
    if ( is_admin() && 'themes.php' == $pagenow && isset( $_GET['activated'] ) ) {
        wp_redirect( admin_url( "themes.php?page=the-conference-pro-license" ) );
    }
}
endif;
add_action( 'admin_init', 'the_conference_pro_redirect_on_activation' );

if( ! function_exists( 'the_conference_pro_update_customizer_search_template' ) ) :
/**
 * Overwrite Customizer Search Template
 **/    
function the_conference_pro_update_customizer_search_template(){
    if( class_exists( 'Customizer_Search' ) ){
        remove_action( 'customize_controls_print_footer_scripts', array( Customizer_Search::instance(), 'footer_scripts' ) );
        add_action( 'customize_controls_print_footer_scripts', function(){ ?>
            <script type="text/html" id="tmpl-search-button">
                <button type="button" class="customize-search-toggle dashicons dashicons-search" aria-expanded="false"><span class="screen-reader-text"><?php _e( 'Search', 'the-conference-pro' ); ?></span></button>
            </script>

            <script type="text/html" id="tmpl-search-form">
                <div id="accordion-section-customizer-search" class="open">
                    <h4 class="customizer-search-section accordion-section-title">
                        <span class="search-input"><?php _e( 'Search', 'the-conference-pro' ); ?></span>
                        <span class="search-field-wrapper">
                            <input type="text" placeholder="<?php _e( 'Search...', 'the-conference-pro' ); ?>" name="customizer-search-input" autofocus="autofocus" id="customizer-search-input" class="customizer-search-input">
                            <button type="button" class="button clear-search" tabindex="0"><?php _e( 'Clear', 'the-conference-pro' ); ?></button>
                        </span>

                    </h4>
                </div>
            </script>
            <?php
        });
    }
}
endif;
add_action( 'init', 'the_conference_pro_update_customizer_search_template' );

if( ! function_exists( 'the_conference_pro_allowed_social_protocols' ) ) :
/* List of allowed social protocols in HTML attributes.
@param array $protocols Array of allowed protocols. @return array
*/
function the_conference_pro_allowed_social_protocols( $protocols ) {
    $social_protocols = array(
        'skype',
        'viber'
    );
return array_merge( $protocols, $social_protocols );
}
endif;
add_filter( 'kses_allowed_protocols' , 'the_conference_pro_allowed_social_protocols' );