<?php
/**
 * The Conference Pro Standalone Functions.
 *
 * @package The_Conference_Pro
 */

if ( ! function_exists( 'the_conference_pro_posted_on' ) ) :
/**
 * Prints HTML with meta information for the current post-date/time.
 */
function the_conference_pro_posted_on() {
	$ed_updated_post_date = get_theme_mod( 'ed_post_update_date', true );
    
    if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
		if( $ed_updated_post_date ){
            $time_string = '<time class="entry-date published updated" datetime="%3$s" itemprop="dateModified">%4$s</time></time><time class="updated" datetime="%1$s" itemprop="datePublished">%2$s</time>';
		}else{
            $time_string = '<time class="entry-date published" datetime="%1$s" itemprop="datePublished">%2$s</time><time class="updated" datetime="%3$s" itemprop="dateModified">%4$s</time>';  
		}        
	}else{
	   $time_string = '<time class="entry-date published updated" datetime="%1$s" itemprop="datePublished">%2$s</time><time class="updated" datetime="%3$s" itemprop="dateModified">%4$s</time>';   
	}

	$time_string = sprintf( $time_string,
		esc_attr( get_the_date( 'c' ) ),
		esc_html( get_the_date() ),
		esc_attr( get_the_modified_date( 'c' ) ),
		esc_html( get_the_modified_date() )
	);
    
    $posted_on = '<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $time_string . '</a>';
	
	echo '<span class="posted-on"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18"><defs><style>.clk{fill:#999;}</style></defs><g transform="translate(-2 -2)"><g transform="translate(2 2)"><path class="clk" d="M11,2a9,9,0,1,0,9,9A9.026,9.026,0,0,0,11,2Zm0,16.2A7.2,7.2,0,1,1,18.2,11,7.21,7.21,0,0,1,11,18.2Z" transform="translate(-2 -2)"/><path class="clk" d="M12.35,7H11v5.4l4.68,2.88.72-1.17-4.05-2.43Z" transform="translate(-2.9 -2.5)"/></g></g></svg>' . $posted_on . '</span>'; // WPCS: XSS OK.

}
endif;

if ( ! function_exists( 'the_conference_pro_posted_by' ) ) :
/**
 * Prints HTML with meta information for the current author.
 */
function the_conference_pro_posted_by( $author_id = '' ) {

    if( $author_id ){
        $author_url = get_author_posts_url( $author_id ); 
        $author_name = get_the_author_meta( 'display_name', $author_id );
    }else{
        $author_url = get_author_posts_url( get_the_author_meta( 'ID' ) );
        $author_name = get_the_author();
    }

	$byline = sprintf(
		'<span class="author" itemprop="name"><a class="url fn n" href="' . esc_url( $author_url ) . '" itemprop="url">' . esc_html( $author_name ) . '</a></span>' 
    );

	echo '<span class="byline" itemprop="author" itemscope itemtype="https://schema.org/Person"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 19 19"><defs><style>.auta{fill:none;}.auta,.autb{stroke:rgba(0,0,0,0);}.autb{fill:#ccc6c8;}</style></defs><g transform="translate(0.5 0.5)"><path class="auta" d="M0,0H18V18H0Z"/><g transform="translate(1.5 1.5)"><path class="autb" d="M9.5,2A7.5,7.5,0,1,0,17,9.5,7.5,7.5,0,0,0,9.5,2ZM5.8,14.21c.322-.675,2.287-1.335,3.7-1.335s3.382.66,3.7,1.335a5.944,5.944,0,0,1-7.395,0Zm8.468-1.088c-1.073-1.3-3.675-1.747-4.77-1.747s-3.7.443-4.77,1.747a6,6,0,1,1,9.54,0Z" transform="translate(-2 -2)"/><path class="autb" d="M11.125,6A2.625,2.625,0,1,0,13.75,8.625,2.618,2.618,0,0,0,11.125,6Zm0,3.75A1.125,1.125,0,1,1,12.25,8.625,1.123,1.123,0,0,1,11.125,9.75Z" transform="translate(-3.625 -3)"/></g></g></svg>' . $byline . '</span>';
}
endif;

if( ! function_exists( 'the_conference_pro_comment_count' ) ) :
/**
 * Comment Count
*/
function the_conference_pro_comment_count(){
    if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
		echo '<span class="comment-box">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 15 15"><defs><style>.cmt{fill:#999;}</style></defs><path class="cmt" d="M15.5,2H3.5A1.5,1.5,0,0,0,2,3.5V17l3-3H15.5A1.5,1.5,0,0,0,17,12.5v-9A1.5,1.5,0,0,0,15.5,2Zm0,10.5H5L3.5,14V3.5h12Z" transform="translate(-2 -2)"/></svg>';
		comments_popup_link(
			sprintf(
				wp_kses(
					/* translators: %s: post title */
					__( 'No Comment<span class="screen-reader-text"> on %s</span>', 'the-conference-pro' ),
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				get_the_title()
			), 
            __( '1 Comment', 'the-conference-pro' ), 
            __( '% Comments', 'the-conference-pro' ) 
		);
		echo '</span>';
	}    
}
endif;

if ( ! function_exists( 'the_conference_pro_category' ) ) :
/**
 * Prints categories
 */
function the_conference_pro_category( $section = '' ){
	// Hide category and tag text for pages.
	if ( 'post' === get_post_type() ) {
		/* translators: used between list items, there is a space after the comma */
		$categories_list = get_the_category();
		if ( $categories_list ) {
            echo '<span class="category">';
                foreach ( $categories_list as $category ) {
                    $cat_name = $category->cat_name;
                    $cat_url = get_term_link( $category->term_id );

                    if( $cat_name && $cat_url ){
                        echo '<a href="'. esc_url( $cat_url ) .'">'. esc_html( $cat_name ) .'</a>';
                    }
                }
            echo '</span>';
		}
	}
}
endif;

if ( ! function_exists( 'the_conference_pro_tag' ) ) :
/**
 * Prints tags
 */
function the_conference_pro_tag(){
	// Hide category and tag text for pages.
	if ( 'post' === get_post_type() ) {
		/* translators: used between list items, there is a space after the comma */
		$post_tags = get_the_tags();
		if ( $post_tags ) {
            echo '<span class="cat-tags"><h5>'. esc_html__( 'TAGS:', 'the-conference-pro' ) .'</h5>';
            foreach( $post_tags as $tag ) {
                echo '<a href="'. esc_url( get_tag_link( $tag->term_id ) ) .'">'. esc_html( $tag->name ) .'</a>'; 
            }
            echo '</span>';
		}
	}
}
endif;

if( ! function_exists( 'the_conference_pro_site_branding' ) ) :
/**
 * Site Branding
*/
function the_conference_pro_site_branding(){ 
    $display_header_text = get_theme_mod( 'header_text', 1 );
    $site_title          = get_bloginfo( 'name', 'display' );
    $description         = get_bloginfo( 'description', 'display' );

    if( ( function_exists( 'has_custom_logo' ) && has_custom_logo() ) && $display_header_text && ( ! empty( $site_title ) || ! empty(  $description  ) ) ){
        $branding_class = ' logo-with-site-identity';                                                            
    } else {
        $branding_class = '';
    } ?>

    <div class="site-branding<?php echo esc_attr( $branding_class ); ?>" itemscope itemtype="https://schema.org/Organization">
		<?php 
            if( function_exists( 'has_custom_logo' ) && has_custom_logo() ){
                echo '<div class="site-logo">';
                the_custom_logo();
                echo '</div><!-- .site-logo -->';
            } 
            
            echo '<div class="site-title-wrap">';

            if( is_front_page() ){ ?>
                <h1 class="site-title" itemprop="name"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home" itemprop="url"><?php bloginfo( 'name' ); ?></a></h1>
                <?php 
            }else{ ?>
                <p class="site-title" itemprop="name"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home" itemprop="url"><?php bloginfo( 'name' ); ?></a></p>
            <?php
            }

            if ( $description || is_customize_preview() ){ ?>
                <p class="site-description" itemprop="description"><?php echo $description; ?></p>
            <?php
            }
        ?>
        </div><!-- .site-title-wrap -->
	</div>    
    <?php
}
endif;

if( ! function_exists( 'the_conference_pro_primary_nagivation' ) ) :
/**
 * Primary Navigation.
*/
function the_conference_pro_primary_nagivation( $mobile = false ){ 
    $enabled_section = array();
    $ed_one_page     = get_theme_mod( 'ed_one_page', false );
    $ed_home_link    = get_theme_mod( 'ed_home_link', true );
    $home_sections   = get_theme_mod( 'front_sort', array( 'about', 'stat-counter', 'recent-conference', 'speakers', 'schedule', 'ticket', 'testimonial', 'timer-cta', 'sponsor', 'cta', 'faq', 'gallery', 'blog', 'contact' ) );
    
    $label_about        = get_theme_mod( 'label_about', __( 'About', 'the-conference-pro' ) );
    $label_stat_counter = get_theme_mod( 'label_stat_counter', __( 'Statictics', 'the-conference-pro' ) );
    $label_conference   = get_theme_mod( 'label_conference', __( 'Conferences', 'the-conference-pro' ) );
    $label_speakers     = get_theme_mod( 'label_speakers', __( 'Speakers', 'the-conference-pro' ) );
    $label_schedule     = get_theme_mod( 'label_schedule', __( 'Schedule', 'the-conference-pro' ) );
    $label_ticket       = get_theme_mod( 'label_ticket', __( 'Ticket', 'the-conference-pro' ) );
    $label_testimonial  = get_theme_mod( 'label_testimonial', __( 'Testimonial', 'the-conference-pro' ) );
    $label_timer_cta    = get_theme_mod( 'label_timer_cta', __( 'Timer CTA', 'the-conference-pro' ) );
    $label_sponsor      = get_theme_mod( 'label_sponsor', __( 'Sponsor', 'the-conference-pro' ) );
    $label_cta          = get_theme_mod( 'label_cta', __( 'Call to Action', 'the-conference-pro' ) );
    $label_faq          = get_theme_mod( 'label_faq', __( 'FAQ\'S', 'the-conference-pro' ) );
    $label_gallery      = get_theme_mod( 'label_gallery', __( 'Gallery', 'the-conference-pro' ) );
    $label_blog         = get_theme_mod( 'label_blog', __( 'Blog', 'the-conference-pro' ) );
    $label_contact      = get_theme_mod( 'label_contact', __( 'Contact', 'the-conference-pro' ) );
    
    $menu_label = array();
    if( ! empty( $label_about ) ) $menu_label['about']                  = $label_about;
    if( ! empty( $label_stat_counter ) ) $menu_label['stat-counter']    = $label_stat_counter;
    if( ! empty( $label_conference ) ) $menu_label['recent-conference'] = $label_conference;
    if( ! empty( $label_speakers ) ) $menu_label['speakers']            = $label_speakers;
    if( ! empty( $label_schedule ) ) $menu_label['schedule']            = $label_schedule;
    if( ! empty( $label_ticket ) ) $menu_label['ticket']                = $label_ticket;
    if( ! empty( $label_testimonial ) ) $menu_label['testimonial']      = $label_testimonial;
    if( ! empty( $label_timer_cta ) ) $menu_label['timer-cta']          = $label_timer_cta;
    if( ! empty( $label_sponsor ) ) $menu_label['sponsor']              = $label_sponsor;
    if( ! empty( $label_cta ) ) $menu_label['cta']                      = $label_cta;
    if( ! empty( $label_faq ) ) $menu_label['faq']                      = $label_faq;
    if( ! empty( $label_gallery ) ) $menu_label['gallery']              = $label_gallery;
    if( ! empty( $label_blog ) ) $menu_label['blog']                    = $label_blog;
    if( ! empty( $label_contact ) ) $menu_label['contact']              = $label_contact;
    
    foreach( $home_sections as $section ){
        if( array_key_exists( $section, $menu_label ) ){
            $enabled_section[] = array(
                'id'    => $section . '_section',
                'label' => $menu_label[$section],
            );
        }
    }

    $nav_id = $mobile ? 'mobile-site-navigation' : 'site-navigation';
    $schema = $mobile ? '' : ' itemscope itemtype="https://schema.org/SiteNavigationElement"'; 
    
    if( $ed_one_page && ( 'page' == get_option( 'show_on_front' ) ) && $enabled_section ){ ?>
        <nav id="<?php echo $nav_id; ?>" class="main-navigation" role="navigation"<?php echo $schema; ?>>
            <button type="button" class="toggle-button">
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
            </button>
            <ul>
                <?php if( $ed_home_link ){ ?>
                <li class="<?php if( is_front_page() ) echo esc_attr( 'current-menu-item' ); ?>"><a href="<?php echo esc_url( home_url( '/' ) . '#banner_section' ); ?>"><?php esc_html_e( 'Home', 'the-conference-pro' ); ?></a></li>
            <?php }
                foreach( $enabled_section as $section ){ 
                    if( $section['label'] ){
            ?>
                    <li><a href="<?php echo esc_url( home_url( '/' ) . '#' . esc_attr( $section['id'] ) ); ?>"><?php echo esc_html( $section['label'] );?></a></li>                        
            <?php 
                    } 
                }
            ?>
            </ul>
        </nav>
        <?php
    }else{
    ?>
    	<nav id="<?php echo $nav_id; ?>" class="main-navigation" role="navigation"<?php echo $schema; ?>>
            <button type="button" class="toggle-button">
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
            </button>
    		<?php
    			wp_nav_menu( array(
    				'theme_location' => 'primary',
    				'menu_id'        => 'primary-menu',
                    'menu_class'     => 'nav-menu', 
                    'fallback_cb'    => 'the_conference_pro_primary_menu_fallback',
    			) );
    		?>
    	</nav><!-- #site-navigation -->
    <?php
    }
}
endif;

if( ! function_exists( 'the_conference_pro_primary_menu_fallback' ) ) :
/**
 * Fallback for primary menu
*/
function the_conference_pro_primary_menu_fallback(){
    if( current_user_can( 'manage_options' ) ){
        echo '<ul id="primary-menu" class="menu">';
        echo '<li><a href="' . esc_url( admin_url( 'nav-menus.php' ) ) . '">' . esc_html__( 'Click here to add a menu', 'the-conference-pro' ) . '</a></li>';
        echo '</ul>';
    }
}
endif;

if( ! function_exists( 'the_conference_pro_breadcrumb' ) ) :
/**
 * Breadcrumbs
*/
function the_conference_pro_breadcrumb(){ 
    global $post;
    $post_page  = get_option( 'page_for_posts' ); //The ID of the page that displays posts.
    $show_front = get_option( 'show_on_front' ); //What to show on the front page    
    $home       = get_theme_mod( 'home_text', __( 'Home', 'the-conference-pro' ) ); // text for the 'Home' link
    $delimiter  = '<i class="fas fa-long-arrow-alt-right"></i>';
    $before     = '<span class="current" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">'; // tag before the current crumb
    $after      = '</span>'; // tag after the current crumb
    
    if( get_theme_mod( 'ed_breadcrumb', true ) ){
        $depth = 1;
        echo '<div class="breadcrumb"><div id="crumbs" itemscope itemtype="https://schema.org/BreadcrumbList"><span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( home_url() ) . '" class="home_crumb"><span itemprop="name">' . esc_html( $home ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
        
        if( is_home() ){
            $depth = 2;
            echo $before . '<a itemprop="item" href="'. esc_url( get_the_permalink() ) .'"><span itemprop="name">' . esc_html( single_post_title( '', false ) ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
            
        }elseif( is_category() ){
            $depth = 2;
            $thisCat = get_category( get_query_var( 'cat' ), false );

            if( $show_front === 'page' && $post_page ){ //If static blog post page is set
                $p = get_post( $post_page );
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_permalink( $post_page ) ) . '"><span itemprop="name">' . esc_html( $p->post_title ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth ++;  
            }

            if ( $thisCat->parent != 0 ) {
                $parent_categories = get_category_parents( $thisCat->parent, false, ',' );
                $parent_categories = explode( ',', $parent_categories );

                foreach ( $parent_categories as $parent_term ) {
                    $parent_obj = get_term_by( 'name', $parent_term, 'category' );
                    if( is_object( $parent_obj ) ){
                        $term_url    = get_term_link( $parent_obj->term_id );
                        $term_name   = $parent_obj->name;
                        echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( $term_url ) . '"><span itemprop="name">' . esc_html( $term_name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                        $depth ++;
                    }
                }
            }
            echo $before . '<a itemprop="item" href="' . esc_url( get_term_link( $thisCat->term_id) ) . '"><span itemprop="name">' .  esc_html( single_cat_title( '', false ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" />' . $after;
        
        }elseif( is_tax( array( 'rara_portfolio_categories', 'rara_schedule_categories', 'rara_hall_categories' ) ) ){
            $depth          = 2;
            $portfolio      = the_conference_pro_get_page_id_by_template( 'templates/portfolio.php' );
            $schedule       = the_conference_pro_get_page_id_by_template( 'templates/schedule.php' );
            $venue          = the_conference_pro_get_page_id_by_template( 'templates/venue.php' );
            $queried_object = get_queried_object();
            $taxonomy       = 'rara_portfolio_categories';

            if( $portfolio && is_tax( 'rara_portfolio_categories' ) ) {

                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $portfolio[0] ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $portfolio[0] ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth++;

            }elseif( $schedule && is_tax( 'rara_schedule_categories' ) ) {

                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $schedule[0] ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $schedule[0] ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth++;

            }elseif ( $venue && is_tax( 'rara_hall_categories' ) ) {
                
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $venue[0] ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $venue[0] ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth++;
                
            }
            if( $queried_object->parent != 0 ) {
                $parent_categories = get_term_parents_list( $queried_object->parent, $taxonomy, array( 'separator' => ',' ) );
                $parent_categories = explode( ',', $parent_categories );
                foreach ( $parent_categories as $parent_term ) {
                    $parent_obj = get_term_by( 'name', $parent_term,$taxonomy );
                    if( is_object( $parent_obj ) ){
                        $term_url    = get_term_link( $parent_obj->term_id );
                        $term_name   = $parent_obj->name;
                        echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( $term_url ) . '"><span itemprop="name">' . esc_html( $term_name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                        $depth ++;
                    }
                }
            }
            //Displaying the current viewed term object 
            echo $before . '<a itemprop="item" href="' . esc_url( get_term_link( $queried_object->term_id ) ) . '"><span itemprop="name">' . esc_html( $queried_object->name ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;

        }elseif(is_woocommerce_activated() && ( is_product_category() || is_product_tag() ) ){ //For Woocommerce archive page
            
            $depth = 2;
            $current_term = $GLOBALS['wp_query']->get_queried_object();
            if( is_product_category() ){
                $ancestors = get_ancestors( $current_term->term_id, 'product_cat' );
                $ancestors = array_reverse( $ancestors );
                foreach ( $ancestors as $ancestor ) {
                    $ancestor = get_term( $ancestor, 'product_cat' );    
                    if ( ! is_wp_error( $ancestor ) && $ancestor ) {
                        echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_term_link( $ancestor ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $ancestor->name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                        $depth ++;
                    }
                }
            }           
            echo $before . '<a itemprop="item" href="' . esc_url( get_term_link( $current_term->term_id ) ) . '"><span itemprop="name">' . esc_html( $current_term->name ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />' . $after;
            
        } elseif(is_woocommerce_activated() && is_shop() ){ //Shop Archive page
            $depth = 2;
            if ( get_option( 'page_on_front' ) == wc_get_page_id( 'shop' ) ) {
                return;
            }
            $_name = wc_get_page_id( 'shop' ) ? get_the_title( wc_get_page_id( 'shop' ) ) : '';
            $shop_url = wc_get_page_id( 'shop' ) && wc_get_page_id( 'shop' ) > 0  ? get_the_permalink( wc_get_page_id( 'shop' ) ) : home_url( '/shop' );
    
            if ( ! $_name ) {
                $product_post_type = get_post_type_object( 'product' );
                $_name = $product_post_type->labels->singular_name;
            }
            echo $before . '<a itemprop="item" href="' . esc_url( $shop_url ) . '"><span itemprop="name">' . esc_html( $_name ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
            
        }elseif( is_tag() ){
            $queried_object = get_queried_object();
            $depth = 2;

            echo $before . '<a itemprop="item" href="' . esc_url( get_term_link( $queried_object->term_id ) ) . '"><span itemprop="name">' . esc_html( single_tag_title( '', false ) ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
     
        }elseif( is_author() ){
            $depth = 2;
            global $author;

            $userdata = get_userdata( $author );
            echo $before . '<a itemprop="item" href="' . esc_url( get_author_posts_url( $author ) ) . '"><span itemprop="name">' . esc_html( $userdata->display_name ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
     
        }elseif( is_search() ){
            $depth = 2;
            $request_uri = $_SERVER['REQUEST_URI'];
            echo $before .'<a itemprop="item" href="'. esc_url( $request_uri ) .'"><span itemprop="name">'. esc_html__( 'Search Results for "', 'the-conference-pro' ) . esc_html( get_search_query() ) . esc_html__( '"', 'the-conference-pro' ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
        
        }elseif( is_day() ){
            $depth = 2;
            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_year_link( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">' . esc_html( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
            $depth ++;
            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_month_link( get_the_time( __( 'Y', 'the-conference-pro' ) ), get_the_time( __( 'm', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">' . esc_html( get_the_time( __( 'F', 'the-conference-pro' ) ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
            $depth ++;
            echo $before .'<a itemprop="item" href="' . esc_url( get_day_link( get_the_time( __( 'Y', 'the-conference-pro' ) ), get_the_time( __( 'm', 'the-conference-pro' ) ), get_the_time( __( 'd', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">'. esc_html( get_the_time( __( 'd', 'the-conference-pro' ) ) ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
        
        }elseif( is_month() ){
            $depth = 2;
            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_year_link( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">' . esc_html( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
            $depth++;
            echo $before .'<a itemprop="item" href="' . esc_url( get_month_link( get_the_time( __( 'Y', 'the-conference-pro' ) ), get_the_time( __( 'm', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">'. esc_html( get_the_time( __( 'F', 'the-conference-pro' ) ) ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
        
        }elseif( is_year() ){
            $depth = 2;
            echo $before .'<a itemprop="item" href="' . esc_url( get_year_link( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) ) . '"><span itemprop="name">'. esc_html( get_the_time( __( 'Y', 'the-conference-pro' ) ) ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
    
        }elseif( is_single() && !is_attachment() ){
            $depth = 2;
            if( is_woocommerce_activated() && 'product' === get_post_type() ){ //For Woocommerce single product
                /** NEED TO CHECK THIS PORTION WHILE INTEGRATION WITH WOOCOMMERCE */
                if ( $terms = wc_get_product_terms( $post->ID, 'product_cat', array( 'orderby' => 'parent', 'order' => 'DESC' ) ) ) {
                    $main_term = apply_filters( 'woocommerce_breadcrumb_main_term', $terms[0], $terms );
                    $ancestors = get_ancestors( $main_term->term_id, 'product_cat' );
                    $ancestors = array_reverse( $ancestors );
                    foreach ( $ancestors as $ancestor ) {
                        $ancestor = get_term( $ancestor, 'product_cat' );    
                        if ( ! is_wp_error( $ancestor ) && $ancestor ) {
                            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_term_link( $ancestor ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $ancestor->name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                            $depth++;
                        }
                    }
                    echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_term_link( $main_term ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $main_term->name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                }
                
                echo $before .'<a href="' . esc_url( get_the_permalink() ) . '" itemprop="item"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
                
            }elseif( get_post_type() == 'rara-portfolio' ){
                $depth     = 2;
                $portfolio = the_conference_pro_get_page_id_by_template( 'templates/portfolio.php' );
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $portfolio[0] ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $portfolio[0] ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth++;

                $cat_object = get_the_terms( get_the_ID(), 'rara_portfolio_categories' );
                $potential_parent = 0;
                
                if( is_array( $cat_object ) ){ 
                    //Now try to find the deepest term of those that we know of
                    $use_term = key( $cat_object );

                    foreach( $cat_object as $key => $object ){
                        //Can't use the next($cat_object) trick since order is unknown
                        if( $object->parent > 0  && ( $potential_parent === 0 || $object->parent === $potential_parent ) ){
                            $use_term = $key;
                            $potential_parent = $object->term_id;
                        }
                    }
                    
                    $cat = $cat_object[$use_term];
                    $cats = get_term_parents_list( $cat, 'rara_portfolio_categories', array( 'separator' => ',' ) );
                    $cats = explode( ',', $cats );

                    foreach ( $cats as $cat ) {
                        $cat_obj = get_term_by( 'name', $cat, 'rara_portfolio_categories' );
                        if( is_object( $cat_obj ) ){
                            $term_url    = get_term_link( $cat_obj->term_id );
                            $term_name   = $cat_obj->name;
                            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( $term_url ) . '"><span itemprop="name">' . esc_html( $term_name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                            $depth ++;
                        }
                    }
                }

                echo $before .'<a href="' . esc_url( get_the_permalink() ) . '" itemprop="item"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;

            }elseif( get_post_type() == 'rara-schedule' ){
                $depth     = 2;
                $schedule = the_conference_pro_get_page_id_by_template( 'templates/schedule.php' );
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $schedule[0] ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $schedule[0] ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                $depth++;

                $cat_object = get_the_terms( get_the_ID(), 'rara_schedule_categories' );
                $potential_parent = 0;
                
                if( is_array( $cat_object ) ){ 
                    //Now try to find the deepest term of those that we know of
                    $use_term = key( $cat_object );

                    foreach( $cat_object as $key => $object ){
                        //Can't use the next($cat_object) trick since order is unknown
                        if( $object->parent > 0  && ( $potential_parent === 0 || $object->parent === $potential_parent ) ){
                            $use_term = $key;
                            $potential_parent = $object->term_id;
                        }
                    }
                    
                    $cat = $cat_object[$use_term];
                    $cats = get_term_parents_list( $cat, 'rara_schedule_categories', array( 'separator' => ',' ) );
                    $cats = explode( ',', $cats );

                    foreach ( $cats as $cat ) {
                        $cat_obj = get_term_by( 'name', $cat, 'rara_schedule_categories' );
                        if( is_object( $cat_obj ) ){
                            $term_url    = get_term_link( $cat_obj->term_id );
                            $term_name   = $cat_obj->name;
                            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( $term_url ) . '"><span itemprop="name">' . esc_html( $term_name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                            $depth ++;
                        }
                    }
                }

                echo $before .'<a href="' . esc_url( get_the_permalink() ) . '" itemprop="item"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;

            }elseif ( get_post_type() != 'post' ){
                $post_type = get_post_type_object( get_post_type() );
                
                if( $post_type->has_archive == true ){// For CPT Archive Link
                   
                   // Add support for a non-standard label of 'archive_title' (special use case).
                   $label = !empty( $post_type->labels->archive_title ) ? $post_type->labels->archive_title : $post_type->labels->name;
                   printf( '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="%1$s" itemprop="item"><span itemprop="name">%2$s</span></a><meta itemprop="position" content="%3$s" />', esc_url( get_post_type_archive_link( get_post_type() ) ), $label, $depth );
                   echo '<span class="separator">' . $delimiter . '</span></span>';
                   $depth ++;
    
                }
                echo $before .'<a href="' . esc_url( get_the_permalink() ) . '" itemprop="item"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
                
            }else{ //For Post
                $cat_object       = get_the_category();
                $potential_parent = 0;
                $depth            = 2;
                
                if( $show_front === 'page' && $post_page ){ //If static blog post page is set
                    $p = get_post( $post_page );
                    echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $post_page ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $p->post_title ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';  
                    $depth++;
                }
                
                if( is_array( $cat_object ) ){ //Getting category hierarchy if any
        
                    //Now try to find the deepest term of those that we know of
                    $use_term = key( $cat_object );
                    foreach( $cat_object as $key => $object )
                    {
                        //Can't use the next($cat_object) trick since order is unknown
                        if( $object->parent > 0  && ( $potential_parent === 0 || $object->parent === $potential_parent ) ){
                            $use_term = $key;
                            $potential_parent = $object->term_id;
                        }
                    }
                    
                    $cat = $cat_object[$use_term];
              
                    $cats = get_category_parents( $cat, false, ',' );
                    $cats = explode( ',', $cats );

                    foreach ( $cats as $cat ) {
                        $cat_obj = get_term_by( 'name', $cat, 'category' );
                        if( is_object( $cat_obj ) ){
                            $term_url    = get_term_link( $cat_obj->term_id );
                            $term_name   = $cat_obj->name;
                            echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a itemprop="item" href="' . esc_url( $term_url ) . '"><span itemprop="name">' . esc_html( $term_name ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /><span class="separator">' . $delimiter . '</span></span>';
                            $depth ++;
                        }
                    }
                }
    
                echo $before .'<a itemprop="item" href="' . esc_url( get_the_permalink() ) . '"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
                
            }
        
        }elseif( !is_single() && !is_page() && get_post_type() != 'post' && !is_404() ){
            $depth = 2;
            $post_type = get_post_type_object(get_post_type());
            if( get_query_var('paged') ){
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_post_type_archive_link( $post_type->name ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $post_type->label ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" />';
                echo ' <span class="separator">' . $delimiter . '</span></span> ' . $before . sprintf( __('Page %s', 'the-conference-pro'), get_query_var('paged') ) . $after;
            }elseif( is_archive() ){
                echo $before .'<a itemprop="item" href="' . esc_url( get_post_type_archive_link( $post_type->name ) ) . '"><span itemprop="name">'. esc_html( $post_type->label ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
            }else{
                echo $before .'<a itemprop="item" href="' . esc_url( get_post_type_archive_link( $post_type->name ) ) . '"><span itemprop="name">'. esc_html( $post_type->label ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
            }
    
        }elseif( is_attachment() ){
            $depth = 2;
            $parent = get_post( $post->post_parent );
            $cat = get_the_category( $parent->ID ); 
            if( $cat ){
                $cat = $cat[0];
                echo get_category_parents( $cat, TRUE, ' <span class="separator">' . $delimiter . '</span> ');
                echo '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $parent ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( $parent->post_title ) . '<span></a><meta itemprop="position" content="'. absint( $depth ).'" />' . ' <span class="separator">' . $delimiter . '</span></span>';
                $depth++;
            }
            echo  $before .'<a itemprop="item" href="' . esc_url( get_the_permalink() ) . '"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
        
        }elseif( is_page() && !$post->post_parent ){
            $depth = 2;
            echo $before .'<a itemprop="item" href="' . esc_url( get_the_permalink() ) . '"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" />'. $after;
    
        }elseif( is_page() && $post->post_parent ){
            $depth = 2;
            $parent_id  = $post->post_parent;
            $breadcrumbs = array();
            
            while( $parent_id ){
                $page = get_post( $parent_id );
                $breadcrumbs[] = '<span itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem"><a href="' . esc_url( get_permalink( $page->ID ) ) . '" itemprop="item"><span itemprop="name">' . esc_html( get_the_title( $page->ID ) ) . '</span></a><meta itemprop="position" content="'. absint( $depth ).'" /></span>';
                $parent_id  = $page->post_parent;
                $depth++;
            }
            $breadcrumbs = array_reverse( $breadcrumbs );
            for ( $i = 0; $i < count( $breadcrumbs) ; $i++ ){
                echo $breadcrumbs[$i];
                if ( $i != count( $breadcrumbs ) - 1 ) echo ' <span class="separator">' . $delimiter . '</span> ';
            }
            echo ' <span class="separator">' . $delimiter . '</span> ' . $before .'<a href="' . get_permalink() . '" itemprop="item"><span itemprop="name">'. esc_html( get_the_title() ) .'</span></a><meta itemprop="position" content="'. absint( $depth ).'" /></span>'. $after;
        
        }elseif( is_404() ){
            echo $before . esc_html__( '404 Error - Page Not Found', 'the-conference-pro' ) . $after;
        }
        
        if( get_query_var('paged') ) echo __( ' (Page', 'the-conference-pro' ) . ' ' . get_query_var('paged') . __( ')', 'the-conference-pro' );
        
        echo '</div></div><!-- .breadcrumb -->';
        
    }                
}
endif;

if( ! function_exists( 'the_conference_pro_theme_comment' ) ) :
/**
 * Callback function for Comment List *
 * 
 * @link https://codex.wordpress.org/Function_Reference/wp_list_comments 
 */
function the_conference_pro_theme_comment( $comment, $args, $depth ){
	if ( 'div' == $args['style'] ) {
		$tag = 'div';
		$add_below = 'comment';
	} else {
		$tag = 'li';
		$add_below = 'div-comment';
	}
?>
	<<?php echo $tag ?> <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ) ?> id="comment-<?php comment_ID() ?>">
	
    <?php if ( 'div' != $args['style'] ) : ?>
    <article id="div-comment-<?php comment_ID() ?>" class="comment-body" itemscope itemtype="https://schema.org/UserComments">
	<?php endif; ?>
    	
        <footer class="comment-meta">
            <div class="comment-author vcard">
        	   <?php 
                    if ( $args['avatar_size'] != 0 ) echo get_avatar( $comment, $args['avatar_size'] );
                    printf( __( '<b class="fn" itemprop="creator" itemscope itemtype="https://schema.org/Person">%s<span class="says">says:</span></b>', 'the-conference-pro' ), get_comment_author_link() ); 
                ?>
            </div><!-- .comment-author vcard -->

            <div class="comment-metadata commentmetadata">
                <?php esc_html_e( 'Posted on', 'the-conference-pro' );?>
                <a href="<?php echo esc_url( htmlspecialchars( get_comment_link( $comment->comment_ID ) ) ); ?>">
                    <time itemprop="commentTime" datetime="<?php echo esc_attr( get_gmt_from_date( get_comment_date() . get_comment_time(), 'Y-m-d H:i:s' ) ); ?>"><?php printf( esc_html__( '%1$s at %2$s', 'the-conference-pro' ), get_comment_date(),  get_comment_time() ); ?></time>
                </a>
            </div>

            <?php if ( $comment->comment_approved == '0' ) : ?>
                <p class="comment-awaiting-moderation"><?php _e( 'Your comment is awaiting moderation.', 'the-conference-pro' ); ?></p>
                <br />
            <?php endif; ?>
        </footer>
        
        <div class="comment-content" itemprop="commentText"><?php comment_text(); ?></div>        
        
         <div class="reply">
            <?php comment_reply_link( array_merge( $args, array( 'add_below' => $add_below, 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
        </div>
        
	<?php if ( 'div' != $args['style'] ) : ?>
    </article><!-- .comment-body -->
	<?php endif; ?>
    
<?php
}
endif;

if( ! function_exists( 'the_conference_pro_sidebar' ) ) :
/**
 * Return sidebar layouts for pages/posts
*/
function the_conference_pro_sidebar( $class = false ){
    global $post;
    $return      = false;

    $show_on_front  = get_option( 'show_on_front' );
    $blogpage_id    = get_option( 'page_for_posts' );
    $frontpage_id   = get_option( 'page_on_front' );
    $home_sections  = the_conference_pro_get_home_sections();
    $general_layout = get_theme_mod( 'layout_style', 'right-sidebar' ); //Default Layout
    
    if( is_front_page() && ! is_home() ){
        if( $home_sections ){
            $return = $class ? '' : false;
        }else{
            $frontpage_layout  = get_post_meta( $frontpage_id, '_conference_sidebar_layout', true );
            $frontpage_layout  = ! empty( $frontpage_layout ) ? $frontpage_layout : 'default-sidebar';
            $frontpage_sidebar = get_post_meta( $frontpage_id, '_conference_sidebar', true );
            $frontpage_sidebar = ! empty( $frontpage_sidebar ) ? $frontpage_sidebar : 'sidebar';

            if( ( $frontpage_layout == 'no-sidebar' ) || ( ( $frontpage_layout == 'default-sidebar' ) && ( $general_layout == 'no-sidebar' ) ) ){
                $return = $class ? 'full-width' : false;
            }elseif( is_active_sidebar( $frontpage_sidebar ) ){
                if( $frontpage_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' : false;
                }elseif( ( $frontpage_layout == 'default-sidebar' && $general_layout == 'right-sidebar' ) || ( $frontpage_layout == 'right-sidebar' ) ){
                    $return = $class ? 'rightsidebar' : $frontpage_sidebar;
                }elseif( ( $frontpage_layout == 'default-sidebar' && $general_layout == 'left-sidebar' ) || ( $frontpage_layout == 'left-sidebar' ) ){
                    $return = $class ? 'leftsidebar' : $frontpage_sidebar;
                }elseif( $frontpage_layout == 'default-sidebar' && $general_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' :  false;
                }elseif( $frontpage_layout == 'default-sidebar' && $general_layout == 'no-sidebar' ){
                    $return = $class ? 'full-width' : false;
                }else{
                    $return = $class ? 'full-width' : false;
                }
            }else{
                $return = $class ? 'full-width' : false;
            }
        }
    }

    if( is_home() ){
        $blog_sidebar = get_theme_mod( 'home_page_sidebar', 'sidebar' );
        
        if( 'page' == $show_on_front && $blogpage_id > 0 ){
            $blogpage_sidebar = get_post_meta( $blogpage_id, '_conference_sidebar', true );
            $blogpage_sidebar = ! empty( $blogpage_sidebar ) ? $blogpage_sidebar : 'default-sidebar';

            $blogpage_layout = get_post_meta( $blogpage_id, '_conference_sidebar_layout', true );
            $blogpage_layout = ! empty( $blogpage_layout ) ? $blogpage_layout : 'default-sidebar';

            if( ( $blogpage_sidebar == 'no-sidebar' ) || ( ( $blogpage_sidebar == 'default-sidebar' ) && ( $general_layout == 'no-sidebar' ) ) ){
                $return = $class ? 'full-width' : false;
            }if( $blogpage_sidebar == 'default-sidebar' && is_active_sidebar( $blog_sidebar ) ){
                if( $blogpage_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' : false;
                }elseif( ( $blogpage_layout == 'default-sidebar' && $general_layout == 'right-sidebar' ) || ( $blogpage_layout == 'right-sidebar' ) ){
                    $return = $class ? 'rightsidebar' : $blog_sidebar;
                }elseif( ( $blogpage_layout == 'default-sidebar' && $general_layout == 'left-sidebar' ) || ( $blogpage_layout == 'left-sidebar' ) ){
                    $return = $class ? 'leftsidebar' : $blog_sidebar;
                }elseif( $blogpage_layout == 'default-sidebar' && $general_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' :  false;
                }elseif( $blogpage_layout == 'default-sidebar' && $general_layout == 'no-sidebar' ){
                    $return = $class ? 'full-width' : false;
                }else{
                    $return = $class ? 'full-width' : false;
                }
            }elseif( is_active_sidebar( $blogpage_sidebar ) ){
                if( $blogpage_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' : false;
                }elseif( ( $blogpage_layout == 'default-sidebar' && $general_layout == 'right-sidebar' ) || ( $blogpage_layout == 'right-sidebar' ) ){
                    $return = $class ? 'rightsidebar' : $blogpage_sidebar;
                }elseif( ( $blogpage_layout == 'default-sidebar' && $general_layout == 'left-sidebar' ) || ( $blogpage_layout == 'left-sidebar' ) ){
                    $return = $class ? 'leftsidebar' : $blogpage_sidebar;
                }elseif( $blogpage_layout == 'default-sidebar' && $general_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' :  false;
                }elseif( $blogpage_layout == 'default-sidebar' && $general_layout == 'no-sidebar' ){
                    $return = $class ? 'full-width' : false;
                }else{
                    $return = $class ? 'full-width' : false;
                }
            }else{
                $return = $class ? 'full-width' : false;
            }

        }elseif( is_active_sidebar( $blog_sidebar ) ){            
            if( $general_layout == 'right-sidebar' ){
                $return = $class ? 'rightsidebar' : $blog_sidebar;
            }elseif( $general_layout == 'left-sidebar' ){
                $return = $class ? 'leftsidebar' : $blog_sidebar;
            }else{
                $return = $class ? 'full-width' : false;
            }
        }else{
            $return = $class ? 'full-width' : false;
        }        
    }

    if( is_archive() ){
        //archive page
        $archive_sidebar = get_theme_mod( 'archive_page_sidebar', 'sidebar' );
        $cat_sidebar     = get_theme_mod( 'cat_archive_page_sidebar', 'default-sidebar' );
        $tag_sidebar     = get_theme_mod( 'tag_archive_page_sidebar', 'default-sidebar' );
        $date_sidebar    = get_theme_mod( 'date_archive_page_sidebar', 'default-sidebar' );
        $author_sidebar  = get_theme_mod( 'author_archive_page_sidebar', 'default-sidebar' ); 
        
        if( is_category() ){
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( $cat_sidebar == 'default-sidebar' && is_active_sidebar( $archive_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    } 
                }else{
                    $return = $archive_sidebar;
                }
            }elseif( is_active_sidebar( $cat_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    } 
                }else{
                    $return = $cat_sidebar;
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }                
        }elseif( is_tag() ){            
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( ( $tag_sidebar == 'default-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $archive_sidebar;
                }
            }elseif( is_active_sidebar( $tag_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $tag_sidebar;
                }           
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }            
        }elseif( is_author() ){            
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( ( $author_sidebar == 'default-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $archive_sidebar;
                }
            }elseif( is_active_sidebar( $author_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $author_sidebar;
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }            
        }elseif( is_date() ){            
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( ( $date_sidebar == 'default-sidebar' && is_active_sidebar( $archive_sidebar ) ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $archive_sidebar;
                }
            }elseif( is_active_sidebar( $date_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $date_sidebar;
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }
        }elseif( is_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() ) ){ //For Woocommerce            
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( is_active_sidebar( 'shop-sidebar' ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                    if( $general_layout == 'left-sidebar' ) $return = 'leftsidebar';
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }       
        }else{
            if( $general_layout == 'no-sidebar' ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( is_active_sidebar( $archive_sidebar ) ){
                if( $class ){
                    if( $general_layout == 'right-sidebar' ){
                        $return = 'rightsidebar';
                    }elseif( $general_layout == 'left-sidebar' ){
                        $return = 'leftsidebar';
                    }else{
                        $return = 'full-width';
                    }
                }else{
                    $return = $archive_sidebar;
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }                      
        }        
    }
    
    if( is_singular() ){         
        $post_sidebar = get_theme_mod( 'single_post_sidebar', 'sidebar' ); //Global sidebar for single post from customizer
        $page_sidebar = get_theme_mod( 'single_page_sidebar', 'sidebar' ); //Global Sidebar for single page from customizer
        $page_layout  = get_theme_mod( 'page_sidebar_layout', 'right-sidebar' ); //Global Layout/Position for Pages
        $post_layout  = get_theme_mod( 'post_sidebar_layout', 'right-sidebar' ); //Global Layout/Position for Posts        
        
        /**
         * Individual post/page layout
        */
        if( get_post_meta( $post->ID, '_conference_sidebar_layout', true ) ){
            $sidebar_layout = get_post_meta( $post->ID, '_conference_sidebar_layout', true );
        }else{
            $sidebar_layout = 'default-sidebar';
        }
        
        /**
         * Individual post/page sidebar
        */     
        if( get_post_meta( $post->ID, '_conference_sidebar', true ) ){
            $single_sidebar = get_post_meta( $post->ID, '_conference_sidebar', true );
        }else{
            $single_sidebar = 'default-sidebar';
        }
        
        if( is_page() ){
            if( is_page_template( array( 'templates/about.php', 'templates/contact.php', 'templates/faq.php', 'templates/ticket.php', 'templates/schedule.php', 'templates/gallery.php', 'templates/speaker.php', 'templates/sponsor.php', 'templates/testimonial.php', 'templates/venue.php' ) ) ){
                $return = $class ? 'full-width' : false; //Fullwidth
            }elseif( $sidebar_layout == 'no-sidebar' || ( $sidebar_layout == 'default-sidebar' && $page_layout == 'no-sidebar' ) ){
                $return = $class ? 'full-width' : false;
            }elseif( $sidebar_layout == 'centered' || ( $sidebar_layout == 'default-sidebar' && $page_layout == 'centered' ) ){
                $return = $class ? 'full-width-centered' : false;
            }elseif( $single_sidebar == 'default-sidebar' && is_active_sidebar( $page_sidebar ) ){
                if( $class ){
                    if( ( $sidebar_layout == 'default-sidebar' && $page_layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) $return = 'rightsidebar';
                    if( ( $sidebar_layout == 'default-sidebar' && $page_layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) $return = 'leftsidebar';
                }else{
                    $return = $page_sidebar;
                }
            }elseif( is_active_sidebar( $single_sidebar ) ){
                if( $class ){
                    if( ( $sidebar_layout == 'default-sidebar' && $page_layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) $return = 'rightsidebar';
                    if( ( $sidebar_layout == 'default-sidebar' && $page_layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) $return = 'leftsidebar';
                }else{
                    $return = $single_sidebar;
                }
            }else{
                $return = $class ? 'full-width' : false; //Fullwidth
            }
        }
        
        if( is_single() ){
            if( 'product' === get_post_type() ){ //For Product Post Type
                if( $post_layout == 'no-sidebar' || $post_layout == 'centered' ){
                    $return = $class ? 'full-width' : false; //Fullwidth
                }elseif( is_active_sidebar( 'shop-sidebar' ) ){
                    if( $class ){
                        if( $post_layout == 'right-sidebar' ) $return = 'rightsidebar'; //With Sidebar
                        if( $post_layout == 'left-sidebar' ) $return = 'leftsidebar';
                    }
                }else{
                    $return = $class ? 'full-width' : false; //Fullwidth
                }
            }elseif( 'post' === get_post_type() ){ //For default post type
                if( $sidebar_layout == 'no-sidebar' || ( $sidebar_layout == 'default-sidebar' && $post_layout == 'no-sidebar' ) ){
                    $return = $class ? 'full-width' : false;
                }elseif( $sidebar_layout == 'centered' || ( $sidebar_layout == 'default-sidebar' && $post_layout == 'centered' ) ){
                    $return = $class ? 'full-width-centered' : false;
                }elseif( $single_sidebar == 'default-sidebar' && is_active_sidebar( $post_sidebar ) ){
                    if( $class ){
                        if( ( $sidebar_layout == 'default-sidebar' && $post_layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) $return = 'rightsidebar';
                        if( ( $sidebar_layout == 'default-sidebar' && $post_layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) $return = 'leftsidebar';
                    }else{
                        $return = $post_sidebar;
                    }
                }elseif( is_active_sidebar( $single_sidebar ) ){
                    if( $class ){
                        if( ( $sidebar_layout == 'default-sidebar' && $post_layout == 'right-sidebar' ) || ( $sidebar_layout == 'right-sidebar' ) ) $return = 'rightsidebar';
                        if( ( $sidebar_layout == 'default-sidebar' && $post_layout == 'left-sidebar' ) || ( $sidebar_layout == 'left-sidebar' ) ) $return = 'leftsidebar';
                    }else{
                        $return = $single_sidebar;
                    }
                }else{
                    $return = $class ? 'full-width' : false; //Fullwidth
                }
            }else{ //Custom Post Type
                if( $post_layout == 'no-sidebar' ){
                    $return = $class ? 'full-width' : false;
                }elseif( $post_layout == 'centered' ){
                    $return = $class ? 'full-width-centered' : false;
                }elseif( is_active_sidebar( $post_sidebar ) ){
                    if( $class ){
                        if( $post_layout == 'right-sidebar' ) $return = 'rightsidebar';
                        if( $post_layout == 'left-sidebar' ) $return = 'leftsidebar';
                    }else{
                        $return = $post_sidebar;
                    }
                }else{
                    $return = $class ? 'full-width' : false; //Fullwidth
                }
            }
        }
    } 
        
    if( is_search() ){
        $search_sidebar = get_theme_mod( 'search_page_sidebar', 'sidebar' );       
        
        if( $general_layout == 'no-sidebar' ){
            $return = $class ? 'full-width' : false;
        }elseif( is_active_sidebar( $search_sidebar ) ){
            if( $class ){
                if( $general_layout == 'right-sidebar' ) $return = 'rightsidebar';
                if( $general_layout == 'left-sidebar' ) $return = 'leftsidebar';
            }else{
                $return = $search_sidebar;
            }
        }else{
            $return = $class ? 'full-width' : false; //Fullwidth
        }        
    }
      
    return $return; 
}
endif;

if( ! function_exists( 'the_conference_pro_get_posts' ) ) :
/**
 * Fuction to list Custom Post Type
*/
function the_conference_pro_get_posts( $post_type = 'post', $slug = false ){    
    $args = array(
    	'posts_per_page'   => -1,
    	'post_type'        => $post_type,
    	'post_status'      => 'publish',
    	'suppress_filters' => true 
    );
    $posts_array = get_posts( $args );
    
    // Initate an empty array
    $post_options = array();
    $post_options[''] = __( ' -- Choose -- ', 'the-conference-pro' );
    if ( ! empty( $posts_array ) ) {
        foreach ( $posts_array as $posts ) {
            if( $slug ){
                $post_options[ $posts->post_title ] = $posts->post_title;
            }else{
                $post_options[ $posts->ID ] = $posts->post_title;    
            }
        }
    }
    return $post_options;
    wp_reset_postdata();
}
endif;

if( ! function_exists( 'the_conference_pro_get_categories' ) ) :
/**
 * Function to list post categories in customizer options
*/
function the_conference_pro_get_categories( $select = true, $taxonomy = 'category', $slug = false ){    
    /* Option list of all categories */
    $categories = array();
    
    $args = array( 
        'hide_empty' => false,
        'taxonomy'   => $taxonomy 
    );
    
    $catlists = get_terms( $args );
    if( $select ) $categories[''] = __( 'Choose Category', 'the-conference-pro' );
    foreach( $catlists as $category ){
        if( $slug ){
            $categories[$category->slug] = $category->name;
        }else{
            $categories[$category->term_id] = $category->name;    
        }        
    }
    
    return $categories;
}
endif;

if( ! function_exists( 'the_conference_pro_get_id_from_page' ) ) :
/**
 * Get page ids from page name.
*/
function the_conference_pro_get_id_from_page( $slider_pages ){
    if( $slider_pages ){
        $ids = array();
        foreach( $slider_pages as $p ){
             if( !empty( $p['page'] ) ){
                $page_ids = get_page_by_title( $p['page'] );
                $ids[] = $page_ids->ID;
             }
        }   
        return $ids;
    }else{
        return false;
    }
}
endif;

if( ! function_exists( 'the_conference_pro_get_patterns' ) ) :
/**
 * Function to list Custom Pattern
*/
function the_conference_pro_get_patterns(){
    $patterns = array();
    $patterns['nobg'] = get_template_directory_uri() . '/images/patterns_thumb/' . 'nobg.png';
    for( $i=0; $i<38; $i++ ){
        $patterns['pattern'.$i] = get_template_directory_uri() . '/images/patterns_thumb/' . 'pattern' . $i .'.png';
    }
    for( $j=1; $j<26; $j++ ){
        $patterns['hbg'.$j] = get_template_directory_uri() . '/images/patterns_thumb/' . 'hbg' . $j . '.png';
    }
    return $patterns;
}
endif;

if( ! function_exists( 'the_conference_pro_get_dynamnic_sidebar' ) ) :
/**
 * Function to list dynamic sidebar
*/
function the_conference_pro_get_dynamnic_sidebar( $nosidebar = false, $sidebar = false, $default = false ){
    $sidebar_arr = array();
    $sidebars = get_theme_mod( 'sidebar' );
    
    if( $default ) $sidebar_arr['default-sidebar'] = __( 'Default Sidebar', 'the-conference-pro' );
    if( $sidebar ) $sidebar_arr['sidebar'] = __( 'Sidebar', 'the-conference-pro' );
    
    if( $sidebars ){        
        foreach( $sidebars as $sidebar ){            
            $id = $sidebar['name'] ? sanitize_title( $sidebar['name'] ) : 'the-conference-pro-sidebar-one';
            $sidebar_arr[$id] = $sidebar['name'];
        }
    }
    
    if( $nosidebar ) $sidebar_arr['no-sidebar'] = __( 'No Sidebar', 'the-conference-pro' );
    
    return $sidebar_arr;
}
endif;

if( ! function_exists( 'the_conference_pro_create_post' ) ) :
/**
 * A function used to programmatically create a post and assign a page template in WordPress. 
 *
 * @link https://tommcfarlin.com/programmatically-create-a-post-in-wordpress/
 * @link https://tommcfarlin.com/programmatically-set-a-wordpress-template/
 */
function the_conference_pro_create_post( $title, $slug, $template ){

	// Setup the author, page
	$author_id = 1;
    
    // Look for the page by the specified title. Set the ID to -1 if it doesn't exist.
    // Otherwise, set it to the page's ID.
    $page = get_page_by_title( $title, OBJECT, 'page' );
    $page_id = ( null == $page ) ? -1 : $page->ID;
    
	// If the page doesn't already exist, then create it
	if( $page_id == -1 ){

		// Set the post ID so that we know the post was created successfully
		$post_id = wp_insert_post(
			array(
				'comment_status' =>	'closed',
				'ping_status'	 =>	'closed',
				'post_author'	 =>	$author_id,
				'post_name'		 =>	$slug,
				'post_title'	 =>	$title,
				'post_status'	 =>	'publish',
				'post_type'		 =>	'page'
			)
		);
        
        if( $post_id ) update_post_meta( $post_id, '_wp_page_template', $template );

	// Otherwise, we'll stop
	}else{
	   update_post_meta( $page_id, '_wp_page_template', $template );
	} // end if

} // end programmatically_create_post
endif;

if( ! function_exists( 'the_conference_pro_get_page_template_url' ) ) :
/**
 * Returns page template url if not found returns home page url
*/
function the_conference_pro_get_page_template_url( $page_template ){
    $args = array(
        'meta_key'   => '_wp_page_template',
        'meta_value' => $page_template,
        'post_type'  => 'page',
        'fields'     => 'ids',
    );
    
    $posts_array = get_posts( $args );
        
    $url = ( $posts_array ) ? get_permalink( $posts_array[0] ) : get_home_url();
    return $url;    
}
endif;

if( ! function_exists( 'the_conference_pro_author_social' ) ) :
/**
 * Author Social Links
*/
function the_conference_pro_author_social(){
    $id      = get_the_author_meta( 'ID' );
    $socials = get_user_meta( $id, '_the_conference_pro_user_social_icons', true );
    $fonts   = array(
        'facebook'     => 'fab fa-facebook-f', 
        'twitter'      => 'fab fa-twitter',
        'instagram'    => 'fab fa-instagram',
        'snapchat'     => 'fab fa-snapchat',
        'pinterest'    => 'fab fa-pinterest',
        'linkedin'     => 'fab fa-linkedin-in',
        'google-plus'  => 'fab fa-google-plus',
        'youtube'      => 'fab fa-youtube'
    );
    
    if( $socials ){
        echo ' <div class="author-social">';
        foreach( $socials as $key => $link ){            
            if( $link ) echo '<a href="' . esc_url( $link ) . '" target="_blank" rel="nofollow noopener" title="'. esc_attr( $key ) .'"><span class="icon"><i class="' . esc_attr( $fonts[$key] ) . '"></i></span></a>';
        }
        echo '</div><!-- .author-social -->';
    }
}
endif;

if( ! function_exists( 'the_conference_pro_get_home_sections' ) ) :
/**
 * Returns Home Sections 
*/
function the_conference_pro_get_home_sections(){
    $home_sections = get_theme_mod( 'front_sort', array( 'about', 'stat-counter', 'recent-conference', 'speakers', 'schedule', 'ticket', 'testimonial', 'timer-cta', 'sponsor', 'cta', 'faq', 'gallery', 'blog', 'contact', 'map' ) );
    
    $enabled_section = array();
        
    foreach( $home_sections as $v ){
        array_push( $enabled_section, $v );
    } 
    
    return apply_filters( 'the_conference_pro_home_sections', $enabled_section );
}
endif;

if( ! function_exists( 'the_conference_pro_escape_text_tags' ) ) :
/**
 * Remove new line tags from string
 *
 * @param $text
 * @return string
 */
function the_conference_pro_escape_text_tags( $text ) {
    return (string) str_replace( array( "\r", "\n" ), '', strip_tags( $text ) );
}
endif;

if( ! function_exists( 'the_conference_pro_fallback_image' ) ) :
/**
 * Prints Fallback Images
*/
function the_conference_pro_fallback_image( $image_size, $id = 0 ){
    $placeholder = get_template_directory_uri() . '/images/fallback/' . $image_size . '.jpg';
    
    if( get_theme_mod( 'ed_lazy_load', true ) && ! is_page_template( 'templates/portfolio.php' ) ){
		$placeholder_src = 'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==';
		$layzr_attr = ' data-layzr="'.esc_attr( $placeholder ).'"';
	} else {
		$placeholder_src = $placeholder;
		$layzr_attr = '';
	}
    
    if( is_int( $id ) ){
        $alt = get_the_title( $id );
    }else{
        $alt = $id;
    }
    
    echo '<img src="' . esc_url( $placeholder_src ) . '" alt="' . esc_attr( strip_tags( $alt ) ) . '"' . $layzr_attr . ' itemprop="image"/>';
}
endif;

if ( ! function_exists( 'the_conference_pro_apply_theme_shortcode' ) ) :
/**
 * Footer Shortcode
*/
function the_conference_pro_apply_theme_shortcode( $string ) {
    if ( empty( $string ) ) {
        return $string;
    }
    $search = array( '[the-year]', '[the-site-link]' );
    $replace = array(
        date_i18n( esc_html__( 'Y', 'the-conference-pro' ) ),
        '<a href="'. esc_url( home_url( '/' ) ) .'">'. esc_html( get_bloginfo( 'name', 'display' ) ) . '</a>',
    );
    $string = str_replace( $search, $replace, $string );
    return $string;
}
endif;

/**
 * Is BlossomThemes Email Newsletters active or not
*/
function is_btnw_activated(){
    return class_exists( 'Blossomthemes_Email_Newsletter' ) ? true : false;        
}

/**
 * Query WooCommerce activation
 */
function is_woocommerce_activated() {
	return class_exists( 'woocommerce' ) ? true : false;
}

/**
 * Check if Contact Form 7 Plugin is installed
*/
function is_cf7_activated(){
    return class_exists( 'WPCF7' ) ? true : false;
}

/**
 * Query Rara theme companion activation
 */
function is_rara_theme_companion_activated() {
    return class_exists( 'Raratheme_Companion_Public' ) ? true : false;
}

/**
 * Query Jetpack activation
*/
function is_jetpack_activated( $gallery = false ){
	if( $gallery ){
        return ( class_exists( 'jetpack' ) && Jetpack::is_module_active( 'tiled-gallery' ) ) ? true : false;
	}else{
        return class_exists( 'jetpack' ) ? true : false;
    }           
}

/**
 * Check if Polylang is active
*/
function is_polylang_active(){
    return class_exists( 'Polylang' ) ? true : false;
}


if( ! function_exists( 'the_conference_pro_get_svg' ) ) :
    /**
     * Return SVG markup.
     *
     * @param array $args {
     *     Parameters needed to display an SVG.
     *
     *     @type string $icon  Required SVG icon filename.
     *     @type string $title Optional SVG title.
     *     @type string $desc  Optional SVG description.
     * }
     * @return string SVG markup.
     */
    function the_conference_pro_get_svg( $args = array() ) {
        // Make sure $args are an array.
        if ( empty( $args ) ) {
            return __( 'Please define default parameters in the form of an array.', 'the-conference-pro' );
        }

        // Define an icon.
        if ( false === array_key_exists( 'icon', $args ) ) {
            return __( 'Please define an SVG icon filename.', 'the-conference-pro' );
        }

        // Set defaults.
        $defaults = array(
            'icon'        => '',
            'title'       => '',
            'desc'        => '',
            'fallback'    => false,
        );

        // Parse args.
        $args = wp_parse_args( $args, $defaults );

        // Set aria hidden.
        $aria_hidden = ' aria-hidden="true"';

        // Set ARIA.
        $aria_labelledby = '';

        /*
         * The Conference Pro doesn't use the SVG title or description attributes; non-decorative icons are described with .screen-reader-text.
         *
         * However, child themes can use the title and description to add information to non-decorative SVG icons to improve accessibility.
         *
         * Example 1 with title: <?php echo the_conference_pro_get_svg( array( 'icon' => 'arrow-right', 'title' => __( 'This is the title', 'textdomain' ) ) ); ?>
         *
         * Example 2 with title and description: <?php echo the_conference_pro_get_svg( array( 'icon' => 'arrow-right', 'title' => __( 'This is the title', 'textdomain' ), 'desc' => __( 'This is the description', 'textdomain' ) ) ); ?>
         *
         * See https://www.paciellogroup.com/blog/2013/12/using-aria-enhance-svg-accessibility/.
         */
        if ( $args['title'] ) {
            $aria_hidden     = '';
            $unique_id       = uniqid();
            $aria_labelledby = ' aria-labelledby="title-' . $unique_id . '"';

            if ( $args['desc'] ) {
                $aria_labelledby = ' aria-labelledby="title-' . $unique_id . ' desc-' . $unique_id . '"';
            }
        }

        // Begin SVG markup.
        $svg = '<svg class="icon icon-' . esc_attr( $args['icon'] ) . '"' . $aria_hidden . $aria_labelledby . ' role="img">';

        // Display the title.
        if ( $args['title'] ) {
            $svg .= '<title id="title-' . $unique_id . '">' . esc_html( $args['title'] ) . '</title>';

            // Display the desc only if the title is already set.
            if ( $args['desc'] ) {
                $svg .= '<desc id="desc-' . $unique_id . '">' . esc_html( $args['desc'] ) . '</desc>';
            }
        }

        /*
         * Display the icon.
         *
         * The whitespace around `<use>` is intentional - it is a work around to a keyboard navigation bug in Safari 10.
         *
         * See https://core.trac.wordpress.org/ticket/38387.
         */
        $svg .= ' <use href="#icon-' . esc_html( $args['icon'] ) . '" xlink:href="#icon-' . esc_html( $args['icon'] ) . '"></use> ';

        // Add some markup to use as a fallback for browsers that do not support SVGs.
        if ( $args['fallback'] ) {
            $svg .= '<span class="svg-fallback icon-' . esc_attr( $args['icon'] ) . '"></span>';
        }

        $svg .= '</svg>';

        return $svg;
    }
endif;

if( ! function_exists( 'the_conference_pro_get_template_page_url' ) ) :
/**
 * Fuction to return page url
*/
function the_conference_pro_get_template_page_url( $template_key = '', $post_type = 'page' ){

    if( ! empty( $template_key ) ){
        $args = array(
            'meta_key'      => '_wp_page_template',
            'meta_value'    => $template_key,
            'post_type'     => $post_type,
        );

        $posts_array = get_posts( $args );

        if ( ! empty( $posts_array ) ) {
            foreach ( $posts_array as $posts ) {
                $post_options[ $posts->ID ] = $posts->ID;    
                $page_template_url = get_permalink( $post_options[ $posts->ID ] );
                return $page_template_url;
            }
        }
    } else {
        return false;
    }
}
endif;

if( ! function_exists( 'the_conference_pro_header_banner' ) ) :
/**
 * Prints header banner
*/
function the_conference_pro_header_banner(){ 
    global $post;

    if( is_home() || is_archive() || is_search() ){
        if( has_header_image() ){
            $image_url = get_header_image();
        }else{
            $image_url = get_template_directory_uri() . '/images/fallback/the-conference-banner-slider.jpg';
        }
    }elseif( is_object( $post ) && in_array( get_post_type(), array( 'post', 'page' ) ) ){
        if( has_post_thumbnail( $post->ID ) ){
            $image_url = get_the_post_thumbnail_url( $post->ID, 'the-conference-banner-slider' );
        }elseif( has_header_image() ){
            $image_url = get_header_image();
        }else{
            $image_url = get_template_directory_uri() . '/images/fallback/the-conference-banner-slider.jpg';
        }
    }elseif( has_header_image() ){
        $image_url = get_header_image();
    }else{
        $image_url = get_template_directory_uri() . '/images/fallback/the-conference-banner-slider.jpg';
    }

    $style = '';

    if( $image_url ){
        $style = 'style="background: url('. esc_url( $image_url ) .') no-repeat;"';
    }
    ?>
    
    <header class="page-header"<?php echo $style; ?>>
        <div class="container">
            <?php 

                if( is_front_page() && is_home() ){
                    $blog_title = get_theme_mod( 'blog_section_title', __( 'Recent Posts', 'the-conference-pro' ) ); 
                    echo '<h2 class="page-title">'. esc_html( $blog_title ) .'</h2>';
                }

                if ( is_home() && ! is_front_page() ){ 
                    echo '<h2 class="page-title">' . single_post_title( '', false ) . '</h2>';
                }

                if( is_archive() ){
                    if( is_author() ){
                        $author_name = get_the_author_meta( 'display_name' );
                        $author_bio  = get_the_author_meta( 'description' );
                        ?>
                        <div class="about-author">
                            <figure class="author-img">
                                <?php echo get_avatar( get_the_author_meta( 'ID' ), 120 ); ?>
                            </figure>
                            <div class="author-content-wrap">
                                <?php 
                                    if( $author_name ){ 
                                        echo '<h3 class="author-name"><span class="title-wrap"><b>'. esc_html__( 'All Posts By :', 'the-conference-pro' ) .'</b> '. esc_html( $author_name ) .'</span>
                                    </h3>';
                                    }

                                    if( $author_bio ){
                                        echo ' <div class="author-info">'. wp_kses_post( get_the_author_meta( 'description' ) ) .'</div>';
                                    }

                                    the_conference_pro_author_social();
                                ?>
                            </div>
                        </div> <!-- .about-author -->
                        <?php 
                    }else{
                        the_archive_title();
                        the_archive_description( '<div class="archive-description">', '</div>' ); 
                    }
                }

                if( is_search() ){ 
                    global $wp_query;
                    echo '<h1 class="page-title">' . esc_html__( 'SEARCH RESULTS FOR:', 'the-conference-pro' ) . '</h1>';
                    get_search_form();
                }
                
                if( is_single() ){
                    $hide_cat_single = get_theme_mod( 'ed_category', false );
                    if( ! $hide_cat_single ) the_conference_pro_category();
                }

                if( is_singular() ){
                    the_title( '<h1 class="page-title">', '</h1>' );
                }

                if( is_single() && 'post' === get_post_type() ){
                    $hide_post_date     = get_theme_mod( 'ed_post_date', false );
                    $hide_post_author   = get_theme_mod( 'ed_post_author', false );
                    $hide_comment_count = get_theme_mod( 'ed_post_comment_count', false );
                    $author_id          = $post->post_author;


                    echo '<div class="entry-meta">';
                    if( ! $hide_post_author ) the_conference_pro_posted_by( $author_id );
                    if( ! $hide_post_date ) the_conference_pro_posted_on();
                    if( ! $hide_comment_count ) the_conference_pro_comment_count();
                    echo '</div>';
                }

                if( is_404() ){
                    echo '<h1 class="page-title">'. esc_html__( 'Error 404', 'the-conference-pro' ) .'</h1>';
                }

                if( ! ( ( is_home() && is_front_page() ) || is_author() || is_search() || ( is_single() && 'post' === get_post_type() ) ) ){
                    the_conference_pro_breadcrumb();
                }
            ?>
        </div>
    </header>
<?php
}
endif;

if( ! function_exists( 'the_conference_pro_get_posts_list' ) ) :
/**
 * Returns Latest, Related Posts
*/
function the_conference_pro_get_posts_list( $status ){
    global $post;
    $related_tax     = get_theme_mod( 'related_taxonomy', 'cat' );

    $args = array(        
        'posts_status'   => 'publish',        
        'posts_per_page' => 3
    );
    
    switch( $status ){
        case 'latest':        
        $title                        = __( 'Latest Articles', 'the-conference-pro' );
        $args['ignore_sticky_posts']  = true;
        $args['post_type']            = 'post';
        break;
        
        case 'related':
        $args['post__not_in']         = array( $post->ID );
        $args['orderby']              = 'rand';
        $args['ignore_sticky_posts']  = true;
        $args['post_type']            = 'post';
        $title                        = get_theme_mod( 'related_post_title', __( 'Recommended Articles', 'the-conference-pro' ) );
        $related_tax                  = get_theme_mod( 'related_taxonomy', 'cat' );
        if( $related_tax == 'cat' ){
            $cats = get_the_category( $post->ID );        
            if( $cats ){
                $c = array();
                foreach( $cats as $cat ){
                    $c[] = $cat->term_id; 
                }
                $args['category__in'] = $c;
            }
        }elseif( $related_tax == 'tag' ){
            $tags = get_the_tags( $post->ID );
            if( $tags ){
                $t = array();
                foreach( $tags as $tag ){
                    $t[] = $tag->term_id;
                }
                $args['tag__in'] = $t;  
            }
        }
        break;
    }
    
    $qry = new WP_Query( $args );
    
    if( $qry->have_posts() ){ ?>    
        <div class="related-post">
            <?php 
            if( $title ) echo '<h3 class="post-title"><span class="title-wrap">' . esc_html( $title ) . '</span></h3>'; 
                echo '<div class="article-wrap">';
                while( $qry->have_posts() ){ $qry->the_post(); ?>
                    <article class="post">
                        <figure class="post-thumbnail">
                            <a href="<?php the_permalink(); ?>">
                                <?php
                                    if( has_post_thumbnail() ){
                                        the_post_thumbnail( 'the-conference-related', array( 'itemprop' => 'image' ) );
                                    }else{ 
                                        the_conference_pro_fallback_image( 'the-conference-related' );
                                    }
                                ?>
                            </a>
                        </figure>
                        <header class="entry-header">
                            <div class="entry-meta">
                                <?php the_conference_pro_posted_on(); ?>
                            </div> 
                            <?php  the_title( '<h4 class="entry-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h4>' ); ?>                       
                        </header>
                    </article>
                <?php }
                echo '</div><!-- .article-wrap -->'
                ?>           
            </div>
        <?php
        wp_reset_postdata();
    }
}
endif;

if( ! function_exists( 'the_conference_pro_number_of_widgets_in_sidebar' ) ) :
/**
 * Count the number of widgets in a sidebar
 * Works for up to ten widgets
 * Usage <?php the_conference_pro_number_of_widgets_in_sidebar( 'promo' ); ?> where promo is the name of the sidebar
 */
function the_conference_pro_number_of_widgets_in_sidebar( $sidebar_name ) {
    global $sidebars_widgets;
    
    $count = 0;
    $class = '';

    if( isset( $sidebars_widgets[$sidebar_name] ) && '' != $sidebars_widgets[$sidebar_name] ){
        $count = count ($sidebars_widgets[$sidebar_name]);
    }

    if( $count > 0 ){
        $class = ' active-widget-' . $count;
    }

    return $class;
}
endif;

if( ! function_exists( 'the_conference_pro_custom_header_link' ) ) :
    /**
     * Additional Link in menu
     */
    function the_conference_pro_custom_header_link(){
        $link_type       = get_theme_mod( 'custom_link_option', 'custom' );
        $label           = get_theme_mod( 'custom_link_label', __( 'BUY TICKET', 'the-conference-pro' ) );
        $link            = get_theme_mod( 'custom_link', '#' );
        $page_id         = get_theme_mod( 'custom_link_page', '' );
        $ed_new_tab      = get_theme_mod( 'ed_custom_link_tab', false );
        $target          = ( $ed_new_tab && 'custom' == $link_type ) ? ' target="_blank"' : '';

        if( 'custom' == $link_type && $link && $label){
            echo '<div class="nav-btn"><a href="' . esc_url( $link ) . '" class="btn custom-link"' . $target . '>' . esc_html( $label ) . '</a></div>';
        }elseif( $page_id ){
            $page_obj = get_post( $page_id );

            if( ! empty( $page_obj ) ){
                $title       = $page_obj->post_title;
                $url         = get_the_permalink( $page_obj->ID );

                echo '<div class="nav-btn"><a href="' . esc_url( $url ) . '" class="btn"' . $target . '>' . esc_html( $title ) . '</a></div>';
            }
        }
    }
endif;

if( ! function_exists( 'the_conference_pro_polylang_language_switcher' ) ) :
    /**
     * Template for Polylang Language Switcher
     */
    function the_conference_pro_polylang_language_switcher(){
        if( is_polylang_active() && has_nav_menu( 'language' ) ){ ?>
            <nav class="language-dropdown">
                <?php
                    wp_nav_menu( array(
                        'theme_location' => 'language',
                        'menu_class'     => 'languages',
                        'fallback_cb'    => false,
                    ) );
                ?>
            </nav><!-- #site-navigation -->
            <?php        
        }
    }
endif;

if( ! function_exists( 'the_conference_pro_get_schedule' ) ) :
    /**
     * Add schedule contents
     */
    function the_conference_pro_get_schedule( $url = '', $readmore = '' ){
       if( is_rara_theme_companion_activated() ){ 
        $time_format = get_theme_mod( 'ed_schedule_24hrs_time', false );
        ?>
            <div class="tab-content-wrap">
                <?php 
                    $schdeule_array = array();
                    $schedule_index = 1;
                    $schedule_terms = get_terms( 
                        array(
                            'taxonomy'   => 'rara_schedule_categories',
                            'hide_empty' => true,
                            'meta_key'   => 'schedule_order',
                            'orderby'    => 'meta_value_num',
                            'order'      => 'ASC'
                    )  );

                    if( $schedule_terms ){
                        echo '<ul class="tab-wrap">';
                        foreach ( $schedule_terms as $term => $value) {
                            $schedule_class = ( $schedule_index === 1 ) ? ' active' : '';
                            $schdeule_array[$schedule_index] = $value->term_id;

                            echo '<li class="tab-'. absint( $schedule_index ).' tab-btn'. esc_attr( $schedule_class ) .'">'. esc_html( $value->name ) .' <span>'. wp_kses_post( $value->description ) .'</span></li>';
                            $schedule_index++;
                        }
                        echo '</ul>';
                    }

                    $hall_array = array();
                    $hall_index = 1;
                    $hall_terms = get_terms( 
                        array(
                            'taxonomy'   => 'rara_hall_categories',
                            'hide_empty' => true,
                            'meta_key'   => 'hall_order',
                            'orderby'    => 'meta_value_num',
                            'order'      => 'ASC'
                    ) );

                    if( $hall_terms ){
                        foreach ( $hall_terms as $term => $value) {
                            $hall_array[$hall_index] = $value->term_id;
                            $hall_index++;
                        }
                    }
                ?>

                <div class="tab-description">
                    <?php 
                        for ( $m=1; $m < $schedule_index; $m++) { 
                            $hall_class = ( $m === 1 ) ? ' active' : ''; ?>
                            <div class="tab-content tab-<?php echo absint( $m ); ?>-content<?php echo esc_attr( $hall_class ); ?>">
                                <?php 
                                    $hall_args = array(
                                        'post_type'      =>'rara-schedule',
                                        'posts_per_page' => -1,
                                        'orderby'        => 'menu_order date',
                                        'order'          =>'ASC',
                                        'status'         =>'publish',
                                    );

                                    if( $schdeule_array && $hall_array ){
                                        $hall_args['tax_query'] = array(
                                            'relation' => 'AND',
                                            array(
                                                'taxonomy' => 'rara_schedule_categories',
                                                'field' => 'term_id',
                                                'terms' => $schdeule_array[$m], //the taxonomy terms I'd like to dynamically query
                                            ),
                                            array(
                                                'taxonomy' => 'rara_hall_categories',
                                                'field'    => 'term_id',
                                                'terms'    => $hall_array,
                                            ),
                                        );
                                    }elseif( $hall_array ){
                                        $hall_args['tax_query'] = array(
                                            array(
                                                'taxonomy' => 'rara_hall_categories',
                                                'field'    => 'term_id',
                                                'terms'    => $hall_array,
                                            ),
                                        );
                                    }elseif( $schdeule_array ){
                                        $hall_args['tax_query'] = array(
                                            array(
                                                'taxonomy' => 'rara_schedule_categories',
                                                'field'    => 'term_id',
                                                'terms'    => $schdeule_array[$m],
                                            ),
                                        );
                                    }

                                    $hall_query = get_posts( $hall_args );

                                    $hall_metas = array(); 
                                    $order      = array();
                                    
                                    foreach ( $hall_query as $hall_post ) {
                                        $hall_taxomomy = get_the_terms( $hall_post->ID, 'rara_hall_categories' );

                                        if( $hall_taxomomy ){
                                            foreach ( $hall_taxomomy as $hall_value ) {
                                                $hall_name  = $hall_value->name;
                                                $hall_id    = $hall_value->term_id;
                                                $hall_order = get_term_meta( $hall_id, 'hall_order', true );

                                                $hall_metas[$hall_id] = array(
                                                    'hall_id'    => $hall_id, 
                                                    'hall_name'  => $hall_name, 
                                                    'hall_order' => $hall_order
                                                );
                                            }
                                        }
                                    }

                                    if( !empty( $hall_metas ) ){
                                        foreach ( $hall_metas as $key => $row ){
                                            $order[$key] = $row['hall_order'];
                                        }
                                        array_multisort( $order, SORT_ASC, $hall_metas);
                                    }

                                    wp_reset_postdata();

                                    $i = 1;

                                    if( $hall_metas ){
                                        echo '<ul class="tab-wrap">';
                                        foreach ( $hall_metas as $name  ) {
                                            $name_class = ( $i === 1 ) ? ' active' : '';

                                            echo '<li class="childTab-'. absint( $i ) .' childTab-btn'. esc_attr( $name_class ) .'">'. esc_html( $name['hall_name'] ) .'</li>';
                                            $i++;
                                        }
                                        echo '</ul>';
                                    }
                                
                                ?>
                                <div class="tab-description">
                                    <?php 

                                        $j = 1;
                                        if( $hall_metas ){
                                            foreach ( $hall_metas as $key => $term_value ) {
                                                $tab_class = ( $j===1 ) ? ' active' : '';

                                                $post_args = array(
                                                    'post_type'      =>'rara-schedule',
                                                    'posts_per_page' => -1,
                                                    'orderby'        => 'menu_order date',
                                                    'order'          =>'ASC',
                                                    'status'         =>'publish',
                                                );


                                                if( $schdeule_array && $hall_array ){
                                                    $post_args['tax_query'] = array(
                                                        'relation' => 'AND',
                                                        array(
                                                            'taxonomy' => 'rara_schedule_categories',
                                                            'field' => 'term_id',
                                                            'terms' => $schdeule_array[$m], //the taxonomy terms I'd like to dynamically query
                                                        ),
                                                        array(
                                                            'taxonomy' => 'rara_hall_categories',
                                                            'field'    => 'term_id',
                                                            'terms'    => $term_value['hall_id'],
                                                        ),
                                                    );
                                                }elseif( $hall_array ){
                                                    $post_args['tax_query'] = array(
                                                        array(
                                                            'taxonomy' => 'rara_hall_categories',
                                                            'field'    => 'term_id',
                                                            'terms'    => $term_value['hall_id'],
                                                        ),
                                                    );
                                                }elseif( $schdeule_array ){
                                                    $post_args['tax_query'] = array(
                                                        array(
                                                            'taxonomy' => 'rara_schedule_categories',
                                                            'field'    => 'term_id',
                                                            'terms'    => $schdeule_array[$m],
                                                        ),
                                                    );
                                                }

                                                $schedule_post = new WP_Query( $post_args );

                                                if( $schedule_post->have_posts() ){
                                                    echo '<div class="childTab-content childTab-'. absint( $j ) .'-content'. esc_attr( $tab_class ) .'">';
                                                    $post_index = 1;
                                                    while( $schedule_post->have_posts() ){
                                                        $schedule_post->the_post();
                                                        $post_class  = ( $post_index === 1 ) ? ' open': '' ;
                                                        $start_time  = get_post_meta( get_the_ID(), '_cp_schedule_start_time', true );
                                                        $end_time    = get_post_meta( get_the_ID(), '_cp_schedule_end_time', true );
                                                        $presentor   = get_post_meta( get_the_ID(), '_cp_schedule_presentor', true );
                                                        $designation = get_post_meta( get_the_ID(), '_cp_schedule_designation', true );

                                                        if ( $time_format ){                                                        
                                                            $start_time  = ! empty( $start_time ) ? date( 'H:i', strtotime( $start_time ) ) : '';
                                                            $end_time    = ! empty( $end_time ) ? date( 'H:i', strtotime( $end_time ) ) : '';
                                                        }else{
                                                            $start_time  = ! empty( $start_time ) ? date( 'h:i A', strtotime( $start_time ) ) : '';
                                                            $end_time    = ! empty( $end_time ) ? date( 'h:i A', strtotime( $end_time ) ) : '';
                                                        }
                                                        ?>
                                                        <div class="toggle-block<?php $post_index; ?>">
                                                            <figure class="toggle-img">
                                                                <?php 
                                                                    if( has_post_thumbnail() ){
                                                                        the_post_thumbnail( 'thumbnail', array( 'itemprop' => 'image' ) );
                                                                    }else{ 
                                                                        the_conference_pro_fallback_image( 'conference-schedule' );
                                                                    }
                                                                ?>
                                                            </figure>
                                                            <div class="toggle-title">
                                                                <?php
                                                                    if( $start_time && $end_time ){
                                                                        $schedule_time = $start_time .' - ' . $end_time;
                                                                    }else{
                                                                        $schedule_time = $start_time;
                                                                    }

                                                                    if( $schedule_time ) echo '<span class="toggle-subtitle">'. esc_html( $schedule_time ).'</span>' ;

                                                                    the_title( '<span class="toggle-maintitle">', '</span>'); 

                                                                    if( $presentor ) echo ' <span class="speaker-name">'. esc_html__( 'By ', 'the-conference-pro' ) . '<b>' .  esc_html( $presentor ) .'</b></span>' ;

                                                                    if( $designation ) echo '<span class="speaker-company">'. esc_html( $designation ) .'</span>' ;
                                                                ?>
                                                            </div>
                                                            <div class="toggle-desc">
                                                                <?php the_content(); ?>
                                                            </div>
                                                        </div>
                                                        <?php
                                                        $post_index++;
                                                    }
                                                    echo '</div>';
                                                    wp_reset_postdata();
                                                }
                                                $j++;
                                            } 
                                        }else{
                                            $post_args = array(
                                                'post_type'      =>'rara-schedule',
                                                'posts_per_page' => -1,
                                                'orderby'        => 'menu_order date',
                                                'order'          =>'ASC',
                                                'status'         =>'publish',
                                                'tax_query' => array(
                                                    array(
                                                        'taxonomy' => 'rara_schedule_categories',
                                                        'field'    => 'term_id',
                                                        'terms'    => $schdeule_array[$m],
                                                    ),
                                                )
                                            );
                                            
                                            $schedule_post = new WP_Query( $post_args );

                                            if( $schedule_post->have_posts() ){
                                                echo '<div class="childTab-content childTab-'. absint( $j ) .'-content'. esc_attr( $tab_class ) .'">';
                                                $post_index = 1;
                                                while( $schedule_post->have_posts() ){
                                                    $schedule_post->the_post();
                                                    $post_class = ( $post_index === 1 ) ? ' open': '' ;

                                                    $start_time  = get_post_meta( get_the_ID(), '_cp_schedule_start_time', true );
                                                    $end_time    = get_post_meta( get_the_ID(), '_cp_schedule_end_time', true );
                                                    $presentor   = get_post_meta( get_the_ID(), '_cp_schedule_presentor', true );
                                                    $designation = get_post_meta( get_the_ID(), '_cp_schedule_designation', true );
                                                    ?>
                                                    <div class="toggle-block<?php $post_index; ?>">
                                                        <figure class="toggle-img">
                                                            <?php 
                                                                if( has_post_thumbnail() ){
                                                                    the_post_thumbnail( 'thumbnail', array( 'itemprop' => 'image' ) );
                                                                }else{ 
                                                                    the_conference_pro_fallback_image( 'conference-schedule' );
                                                                }
                                                            ?>
                                                        </figure>
                                                        <div class="toggle-title">
                                                            <?php
                                                                if( $start_time && $end_time ){
                                                                    $schedule_time = $start_time .' - ' . $end_time;
                                                                }else{
                                                                    $schedule_time = $start_time;
                                                                }

                                                                if( $schedule_time ) echo '<span class="toggle-subtitle">'. esc_html( $schedule_time ).'</span>' ;

                                                                the_title( '<span class="toggle-maintitle">', '</span>'); 

                                                                if( $presentor ) echo ' <span class="speaker-name">'. esc_html__( 'By ', 'the-conference-pro' ) . '<b>' . esc_html( $presentor ) .'</b></span>' ;

                                                                if( $designation ) echo '<span class="speaker-company">'. esc_html( $designation ) .'</span>' ;
                                                            ?>
                                                        </div>
                                                        <div class="toggle-desc">
                                                            <?php the_content(); ?>
                                                        </div>
                                                    </div>
                                                    <?php
                                                    $post_index++;
                                                }
                                                echo '</div>';
                                                wp_reset_postdata();
                                            }
                                            $j++;

                                        }
                                    ?>
                                </div>
                            </div>
                    <?php } ?>
                </div>
                <?php 
                    if( $url && $readmore ){
                        echo '<div class="btn-wrap"><a href="'. esc_url( $url ) .'" class="btn-filled">'. esc_html( $readmore ) .'</a></div>';
                    } 
                ?>
            </div><!-- .tab-content-wrap -->
        <?php }
    }
endif;

if( ! function_exists( 'wp_body_open' ) ) :
/**
 * Fire the wp_body_open action.
 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
*/
function wp_body_open() {
	/**
	 * Triggered after the opening <body> tag.
    */
	do_action( 'wp_body_open' );
}
endif;

if( ! function_exists( 'the_conference_pro_get_page_id_by_template' ) ) :
/**
 * Returns Page ID by Page Template
*/
function the_conference_pro_get_page_id_by_template( $template_name ){
    $args = array(
        'meta_key'   => '_wp_page_template',
        'meta_value' => $template_name
    );
    return $pages = get_pages( $args );    
}
endif;

if( ! function_exists( 'the_conference_pro_get_timezone_identifiers' ) ) :
/**
 * List of all Time Zones
**/    
function the_conference_pro_get_timezone_identifiers(){
    $final = array();    
    $array = timezone_identifiers_list();
    foreach( $array as $value ){
        $final[$value] = $value;
    }
    return $final;
}
endif;